package com.pangding.web.authority.controller;

import com.pangding.web.authority.service.CompanyService;
import com.pangding.web.authority.vo.reqvo.*;
import com.pangding.web.authority.vo.resvo.CompanyLoginResVo;
import com.pangding.web.authority.vo.resvo.GetCompanyDivideResVo;
import com.pangding.web.authority.vo.resvo.GetCompanyResVo;
import com.pangding.web.authority.vo.resvo.SaveCompanyResVo;
import com.pangding.web.vo.system.pd.company.CompanyBankCardVo;
import com.pangding.web.vo.system.pd.company.CompanyVo;
import com.pangding.web.vo.system.req.company.CompanyBankCardReqVo;
import com.pangding.web.vo.system.req.company.CompanyReqVo;
import com.pangding.web.vo.system.res.company.CompanyResVo;
import com.yanzuoguang.util.helper.CheckerHelper;
import com.yanzuoguang.util.vo.PageSizeData;
import com.yanzuoguang.util.vo.ResponseResult;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RestController;

import java.util.List;

/**
 * 胖丁我的管理Controller
 * @Author zhangjinyao
 * @create 2019/6/2 14:48
 */
@RestController
@RequestMapping(value="/pangding")
public class PangdingManagementController {
    @Autowired
    CompanyService companyServiceImpl;

    /**y
     * 分页查询公司信息
     * @param reqVo 继承了分页对象
     * @return 公司列表分页对象
     */
    @RequestMapping(value = "/list", method = RequestMethod.POST, produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseResult<PageSizeData<CompanyResVo>> list(@RequestBody ListCompanyReqVo reqVo){
        return ResponseResult.result(companyServiceImpl.list(reqVo));
    }

    /**
     * 胖丁查询用户，公司和公司扩展表信息以及银行卡列表
     * @param reqVo 包含公司id的请求对象
     * @return 封装了用户，公司，公司扩展表和银行卡列表信息的对象
     */
    @RequestMapping(value = "/queryCompanyDetail", method = RequestMethod.POST, produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseResult<CompanyResVo> queryCompanyDetail(@RequestBody CompanyReqVo reqVo){
        CheckerHelper.newInstance()
                .notBlankCheck("id",reqVo.getId())
                .checkException();
        return ResponseResult.result(companyServiceImpl.companyDetail(reqVo));
    }

    /**y
     * 胖丁查询用户，公司和公司扩展表信息以及银行卡列表
     * @param reqVo 包含公司id的请求对象
     * @return 封装了用户，公司，公司扩展表和银行卡列表信息的对象
     */
    @RequestMapping(value = "/pdDetail", method = RequestMethod.POST, produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseResult<CompanyLoginResVo> pdGetDetail(@RequestBody CompanyGetReqVo reqVo){
        CheckerHelper.newInstance()
                .notBlankCheck("id",reqVo.getId())
                .checkException();
        return ResponseResult.result(companyServiceImpl.pdGetDetail(reqVo));
    }

    /**
     * 根据请求的数据返回对应的供应商或是接口商列表
     * @param reqVo
     * @return
     */
    @RequestMapping(value = "/productOrInterface", method = RequestMethod.POST, produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseResult productOrInterface(@RequestBody ProductOrInterfaceReqVo reqVo){
        return ResponseResult.result(companyServiceImpl.productOrInterface(reqVo));
    }

    /**y
     * 新增用户表（密码为123456），公司主表，公司扩展表，商户表，银行卡表（企业类型有，个人类型没有）
     * @param reqVo 公司以及公司扩展表信息
     * @return 公司信息
     */
    @RequestMapping(value = "/saveCompany", method = RequestMethod.POST, produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseResult<SaveCompanyResVo> saveCompany(@RequestBody OldSaveCompanyReqVo reqVo){
        checkCompany(reqVo);
        return ResponseResult.result(companyServiceImpl.saveCompany(reqVo));
    }

    private void checkCompany(OldSaveCompanyReqVo reqVo){
        CheckerHelper.newInstance()
                .notBlankCheck("account",reqVo.getAccount())
                .notBlankCheck("companyType",reqVo.getCompanyType())
                .notBlankCheck("companyAttribute",reqVo.getCompanyAttribute())
                .notBlankCheck("merchantName",reqVo.getMerchantName())
                .notBlankCheck("merchantAddress",reqVo.getMerchantAddress())
                .notBlankCheck("companyName",reqVo.getCompanyName())
                .notBlankCheck("companyAddress",reqVo.getCompanyAddress())
                .notBlankCheck("businessLicenseCode",reqVo.getBusinessLicenseCode())
                .notBlankCheck("contactName",reqVo.getContactName())
                .notBlankCheck("contactCode",reqVo.getContactCode())
                .checkIdCard("contactCode",reqVo.getContactCode())
                .notBlankCheck("contactPhone",reqVo.getContactPhone())
                .checkPhoneNo("contactPhone",reqVo.getContactPhone())
                .checkException();
    }

    /**
     * 添加公司提交按钮
     * @param reqVo 封装了公司id和验证码的请求对象
     * @return
     */
    @RequestMapping(value = "/companySubmit", method = RequestMethod.POST, produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseResult<SaveCompanyResVo> companySubmit(@RequestBody OldSaveCompanyReqVo reqVo){
        checkCompany(reqVo);
        CheckerHelper.newInstance()
                .notBlankCheck("companyId",reqVo.getCompanyId())
                .checkException();
        return ResponseResult.result(companyServiceImpl.companySubmit(reqVo));
    }

    /**y
     * 我的二维码页面
     * @param reqVo 公司扩展表请求对象
     * @return 公司主表信息
     */
    @RequestMapping(value = "/myQrCodeUrl", method = RequestMethod.POST, produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseResult<CompanyVo> myQrCodeUrl(@RequestBody CompanyGetReqVo reqVo){
        CheckerHelper.newInstance()
                .notBlankCheck("companyId",reqVo.getId())
                .checkException();
        return ResponseResult.result(companyServiceImpl.myQrCodeUrl(reqVo));
    }

    /**
     * 获取二维码url，有则直接返回，没有则调用微信服务生成url
     * @param reqVo 封装了公司id，页面选项的请求对象
     * @return 二维码url
     */
    @RequestMapping(value = "/generateQrCodeUrl", method = RequestMethod.POST, produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseResult<String> generateQrCodeUrl(@RequestBody GenerateQrCodeUrlReqVo reqVo){
        CheckerHelper.newInstance()
                .notBlankCheck("companyId",reqVo.getCompanyId())
                .notBlankCheck("select",reqVo.getSelect())
                .checkException();
        return ResponseResult.result(companyServiceImpl.generateQrCodeUrl(reqVo));
    }

    /**y
     * 保存/修改银行卡信息，并调用第三方接口发送手机验证码
     *
     * @param reqVo 公司银行卡表信息
     * @return 银行卡表主键
     */
    @RequestMapping(value = "/saveBank", method = RequestMethod.POST, produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseResult<CompanyBankCardVo> saveBank(@RequestBody CompanyBankCardReqVo reqVo){
        checkBank(reqVo);
        return companyServiceImpl.saveBank(reqVo);
    }

    private void checkBank(CompanyBankCardReqVo reqVo){
        CheckerHelper.newInstance()
                .notBlankCheck("bankCardCode",reqVo.getBankCardCode())
                .notNumberCheck("bankCardCode",reqVo.getBankCardCode())
                .notBlankCheck("bankName",reqVo.getBankName())
                .notBlankCheck("name",reqVo.getName())
                .notBlankCheck("idCard",reqVo.getIdCard())
                .notBlankCheck("companyId",reqVo.getCompanyId())
                .checkIdCard("idCard",reqVo.getIdCard())
                .notBlankCheck("phone",reqVo.getPhone())
                .checkPhoneNo("phone",reqVo.getPhone())
                .checkException();
    }

    /**y
     * 提交银行卡信息，验证手机验证码，并更新银行卡状态
     * @param reqVo 银行卡表数据+验证码
     * @return 目前返回验证码
     */
    @RequestMapping(value = "/bankSubmit", method = RequestMethod.POST, produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseResult<CompanyBankCardVo> bankSubmit(@RequestBody BankSubmitReqVo reqVo){
        CheckerHelper.newInstance()
                .notBlankCheck("bankCardId",reqVo.getId())
                .notBlankCheck("verifyCode",reqVo.getVerifyCode())
                .checkException();
        return ResponseResult.result(companyServiceImpl.bankSubmit(reqVo));
    }

    /**
     * 获取银行卡列表
     * @param reqVo 继承了分页对象，含公司id
     * @return 银行卡列表分页对象
     */
    @RequestMapping(value = "/bankCardList", method = RequestMethod.POST, produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseResult bankCardList(@RequestBody BankCardListReqVo reqVo){
        CheckerHelper.newInstance()
                .notBlankCheck("companyId",reqVo.getCompanyId())
                .checkException();
        return ResponseResult.result(companyServiceImpl.getBankCardList(reqVo));
    }

    /**y
     * 添加用户
     * @param reqVo 封装了用户信息和用户角色信息的请求对象
     * @return 用户id
     */
    @RequestMapping(value = "/saveUser", method = RequestMethod.POST, produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseResult<String> saveUser(@RequestBody WebUserReqVo reqVo) throws Exception {
        CheckerHelper.newInstance()
                .notBlankCheck("account",reqVo.getAccount())
                .notBlankCheck("password",reqVo.getPassword())
                .notBlankCheck("phone",reqVo.getPhone())
                .checkPhoneNo("phone",reqVo.getPhone())
                .notBlankCheck("confirmPassword",reqVo.getConfirmPassword())
                .notBlankCheck("name",reqVo.getName())
                .notBlankCheck("email",reqVo.getEmail())
                .notBlankCheck("createId",reqVo.getCreateId())
                .notBlankCheck("createName",reqVo.getCreateName())
                .notBlankCheck("companyId",reqVo.getCompanyId())
                .checkException();
        return ResponseResult.result(companyServiceImpl.saveUser(reqVo));
    }

    /**
     * 修改公司审核状态
     * @param reqVo 封装了公司id 的请求对象
     * @return
     */
    @RequestMapping(value = "/updateExamine", method = RequestMethod.POST, produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseResult updateExamine(@RequestBody UpdateExamineReqVo reqVo){
        CheckerHelper.newInstance()
                .notBlankCheck("companyId",reqVo.getCompanyId())
                .checkException();
        companyServiceImpl.updateExamine(reqVo);
        return new ResponseResult();
    }

    /**
     * 删除银行卡
     * @param reqVo 封装了银行卡id的请求对象
     * @return
     */
    @RequestMapping(value = "/deleteBank", method = RequestMethod.POST, produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseResult deleteBank(@RequestBody CompanyBankCardVo reqVo){
        CheckerHelper.newInstance()
                .notBlankCheck("companyBankCardId",reqVo.getId())
                .checkException();
        companyServiceImpl.deleteBank(reqVo);
        return new ResponseResult();
    }

    /**
     * 设置该银行卡为默认支付，并取消其他银行卡的默认
     * @param reqVo 封装了银行卡id的请求对象
     * @return
     */
    @RequestMapping(value = "/setDefault", method = RequestMethod.POST, produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseResult setDefault(@RequestBody CompanyBankCardVo reqVo){
        CheckerHelper.newInstance()
                .notBlankCheck("bankCardId",reqVo.getId())
                .checkException();
        companyServiceImpl.setDefault(reqVo);
        return new ResponseResult();
    }

    /**
     * 批量删除用户
     * @param reqVo 封装了用户id列表的请求对象
     * @return
     */
    @RequestMapping(value = "/batchDelete", method = RequestMethod.POST, produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseResult batchDelete(@RequestBody BatchDeleteReqVo reqVo){
        CheckerHelper.newInstance()
                .notBlankCheck("userIdList",reqVo.getUserIdList())
                .notBlankCheck("roleId",reqVo.getRoleId())
                .checkException();
        companyServiceImpl.batchDelete(reqVo);
        return new ResponseResult();
    }

    /**
     * 通过公司id查询该公司邀请的公司
     * @param reqVo 封装了公司id的请求对象
     * @return
     */
    @RequestMapping(value = "/myInvite", method = RequestMethod.POST, produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseResult<PageSizeData<CompanyResVo>> myInvite(@RequestBody ListCompanyReqVo reqVo){
        CheckerHelper.newInstance()
                .notBlankCheck("companyId",reqVo.getCompanyId())
                .checkException();
        return ResponseResult.result(companyServiceImpl.myInvite(reqVo));
    }

    /**
     * 根据二维码源码获取公司数据
     * @param reqVo 封装了二维码源码的请求对象
     * @return
     */
    @RequestMapping(value = "/getCompanyByQrCode",method = RequestMethod.POST,produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseResult<GetCompanyResVo> getCompanyByQrCode(@RequestBody GetCompanyByQrCodeReqVo reqVo){
        CheckerHelper.newInstance()
                .notBlankCheck("qrCode",reqVo.getQrCode())
                .checkException();
        return ResponseResult.resultAllowNull(companyServiceImpl.getCompanyByQrCode(reqVo));
    }

    @RequestMapping(value = "/findCompanyDivideInfo",method = RequestMethod.POST,produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseResult<List<GetCompanyDivideResVo>> findCompanyDivideInfo(@RequestBody CompanyDivideReqVo req){
       CheckerHelper.newInstance().notBlankCheck("companyList", req.getCompanyIdList()).checkException();
       return ResponseResult.result(companyServiceImpl.findCompanyDivideInfo(req));
    }

    /**
     * 审核公司
     * @param reqVo
     * @return
     */
    @RequestMapping(value = "/examineConfig",method = RequestMethod.POST,produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseResult examineConfig(@RequestBody ExamineConfigReqVo reqVo){
        companyServiceImpl.examineConfig(reqVo);
        return new ResponseResult();
    }

//    @RequestMapping(value = "updateCompanyMoney",produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
//    public  ResponseResult<String> updateCompanyMoney(@RequestBody CompanyMoneyUpdateReqVo req){
//        CheckerHelper.newInstance()
//                .notBlankCheck("companyId",req.getCompanyId())
//                .notBlankCheck("money", req.getMoney())
//                .notBlankCheck("moneyType", req.getMoney())
//                .checkException();
//        return companyServiceImpl.updateCompanyMoney(req);
//    }
}
