package com.pangding.web.authority.service.company.impl;

import com.pangding.web.authority.dao.company.CompanyBankCardDao;
import com.pangding.web.authority.dao.company.CompanyDao;
import com.pangding.web.authority.feign.PayFeign;
import com.pangding.web.authority.service.company.CompanyBankCardService;
import com.pangding.web.authority.util.CompanyBankCardConstant;
import com.pangding.web.authority.vo.reqvo.CompanyBankCardGetReqVo;
import com.pangding.web.vo.system.pd.company.CompanyBankCardVo;
import com.pangding.web.vo.system.pd.company.CompanyVo;
import com.pangding.web.vo.system.req.company.CompanyBankCardReqVo;
import com.pangding.web.vo.system.res.company.CompanyBankCardResVo;
import com.yanzuoguang.util.base.ObjectHelper;
import com.yanzuoguang.util.exception.CodeException;
import com.yanzuoguang.util.helper.DateHelper;
import com.yanzuoguang.util.helper.StringHelper;
import com.yanzuoguang.util.vo.PageSizeData;
import com.yanzuoguang.util.vo.ResponseResult;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.Date;
import java.util.List;

@Service
public class CompanyBankCardServiceImpl implements CompanyBankCardService {

    @Autowired
    private CompanyDao companyDao;
    @Autowired
    private CompanyBankCardDao companyBankCardDao;
    @Autowired
    private PayFeign payFeign;

    @Override
    public PageSizeData<CompanyBankCardResVo> findCompanyBankCardListPage(CompanyBankCardReqVo req){
        return companyBankCardDao.getBankCardListPage(req);
    }

    @Override
    public List<CompanyBankCardResVo> findCompanyBankCardList(CompanyBankCardReqVo req){
        return companyBankCardDao.getBankCardList(req);
    }

    /**
     * 保存/更新银行卡信息，并调用第三方接口发送手机验证码
     *
     * @param reqVo 公司银行卡表信息
     * @return 银行卡表主键
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public ResponseResult<CompanyBankCardVo> saveBank(CompanyBankCardReqVo reqVo) {
        // 公司是否存在
        CompanyVo companyVo = companyDao.load(reqVo.getCompanyId(), CompanyVo.class);
        if (StringHelper.isEmpty(companyVo)) {
            throw new CodeException("用户信息没有找到。");
        }
        // 判断银行卡是否存在
        Date date = DateHelper.getDateTime(companyVo.getCreateTime());
        String bankId = StringHelper.getNewIdMD5(date, companyVo.getId(), reqVo.getBankCardCode());
        CompanyBankCardVo companyBankCardVo = companyBankCardDao.load(bankId, CompanyBankCardVo.class);
        // 银行卡不为空，并且状态为已绑定成功
        if (!StringHelper.isEmpty(companyBankCardVo)) {
            if (com.pangding.web.constant.CompanyConstant.BANK_STATUS_SUCCESS == companyBankCardVo.getStatus()) {
                throw new CodeException("银行卡已绑定，不可修改。");
            }
            //修改数据库
            reqVo.setReservedPhone(reqVo.getPhone());
            ObjectHelper.writeWithFrom(companyBankCardVo, reqVo);
            companyBankCardDao.update(companyBankCardVo);
        } else {
            companyBankCardVo = new CompanyBankCardVo();
            reqVo.setId(bankId);
            reqVo.setStatus(0);
            reqVo.setType(0);
            reqVo.setReservedPhone(reqVo.getPhone());
            // 查询该公司是否已经有默认银行卡了
            CompanyBankCardReqVo companyBankCardReqVo = new CompanyBankCardReqVo();
            companyBankCardReqVo.setCompanyId(reqVo.getCompanyId());
            companyBankCardReqVo.setIsDefault(CompanyBankCardConstant.IS_DEFAULT);
            CompanyBankCardResVo companyBankCardResVo = companyBankCardDao.load(companyBankCardReqVo, CompanyBankCardResVo.class);
            //若该银行卡是第一张银行卡，则设为默认
            if (StringHelper.isEmpty(companyBankCardResVo)) {
                reqVo.setIsDefault(CompanyBankCardConstant.IS_DEFAULT);
            } else {
                reqVo.setIsDefault(CompanyBankCardConstant.IS_NOT_DEFAULT);
            }
            ObjectHelper.writeWithFrom(companyBankCardVo, reqVo);
            companyBankCardDao.create(companyBankCardVo);
        }
        ResponseResult<CompanyBankCardReqVo> result = payFeign.bindBack(reqVo);
        if (StringHelper.isEmpty(result) || !"00".equals(result.getCode())) {
            throw new CodeException("绑定银行卡失败，请重试");
        }
        CompanyBankCardReqVo resVo = result.getData();
        ObjectHelper.writeWithFrom(companyBankCardVo, resVo);
        companyBankCardDao.update(companyBankCardVo);
        return ResponseResult.result(companyBankCardVo);
    }

//    /**
//     * 提交银行卡信息，验证手机验证码，并更新银行卡状态
//     *
//     * @param reqVo 银行卡表数据+验证码
//     * @return 目前返回验证码
//     */
//    @Override
//    @Transactional(rollbackFor = Exception.class)
//    public CompanyBankCardVo bankSubmit(BankSubmitReqVo reqVo) {
//        /*根据银行卡id获取银行卡信息*/
//        CompanyBankCardGetReqVo getReqVo = new CompanyBankCardGetReqVo();
//        getReqVo.setId(reqVo.getId());
//        CompanyBankCardVo companyBankCardVo = companyBankCardDao.load(getReqVo, CompanyBankCardVo.class);
//
//        /*此处应该将companyBankCardVo和verifyCode发送到第三方接口，获取到的返回值用于更新银行卡状态*/
//        CompanyBankCardReqVo companyBankCardReqVo = new CompanyBankCardReqVo();
//        ObjectHelper.writeWithFrom(companyBankCardReqVo, companyBankCardVo);
//        companyBankCardReqVo.setPhone(companyBankCardVo.getReservedPhone());
//        companyBankCardReqVo.setValidateCode(reqVo.getVerifyCode());
//        ResponseResult<CompanyBankCardReqVo> result = payFeign.bindBack(companyBankCardReqVo);
//        if (null == result || !"00".equals(result.getCode())) {
//            throw new CodeException("绑定银行卡失败，请重试");
//        }
//        CompanyBankCardReqVo resVo = result.getData();
//        ObjectHelper.writeWithFrom(companyBankCardVo, resVo);
//        companyBankCardDao.update(companyBankCardVo);
//
//        return companyBankCardVo;
//    }

    /**
     * 调用支付服务解绑银行卡，并更新银行卡状态
     *
     * @param req 封装了银行卡id的请求对象
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteBank(CompanyBankCardReqVo req) {
        // 查询银行卡信息
        CompanyBankCardVo bankCardVo = companyBankCardDao.load(req.getId(), CompanyBankCardVo.class);
        if(StringHelper.isEmpty(bankCardVo)){
            throw new CodeException("银行卡未找到。");
        }
        // 解绑通联绑定的银行卡
        CompanyBankCardReqVo companyBankCardReqVo = new CompanyBankCardReqVo();
        ObjectHelper.writeWithFrom(companyBankCardReqVo, bankCardVo);
        ResponseResult result = payFeign.unbindBankCard(companyBankCardReqVo);
        if (StringHelper.isEmpty(result) || !"00".equals(result.getCode())) {
            throw new CodeException("解绑失败，请重试");
        }
        // 若删除的银行卡为默认银行卡，则需要设置其他一张卡为默认银行卡
        int isDefault = bankCardVo.getIsDefault();
        int removeFlag = companyBankCardDao.remove(bankCardVo);
        if(removeFlag > 0){
            if (CompanyBankCardConstant.IS_DEFAULT == isDefault) {
                CompanyBankCardGetReqVo getReqVo = new CompanyBankCardGetReqVo();
                getReqVo.setCompanyId(bankCardVo.getCompanyId());
                CompanyBankCardVo companyBankCardVo = companyBankCardDao.load(getReqVo, CompanyBankCardVo.class);
                if (StringHelper.isEmpty(companyBankCardVo)) {
                    companyBankCardVo.setIsDefault(CompanyBankCardConstant.IS_DEFAULT);
                    companyBankCardDao.update(companyBankCardVo);
                }
            }
        }
    }

    /**
     * 设置该银行卡为默认支付，并取消其他银行卡的默认
     *
     * @param req 封装了银行卡id的请求对象
     * @return
     */
    @Override
    public void setDefaultBankCard(CompanyBankCardReqVo req) {
        // 查询银行卡
        CompanyBankCardVo companyBankCardVo = companyBankCardDao.load(req, CompanyBankCardVo.class);
        if(StringHelper.isEmpty(companyBankCardVo)){
            throw new CodeException("银行卡未找到。");
        }
        // 将该公司下所有银行卡设为非默认
        CompanyBankCardReqVo updateDefaultReqVo = new CompanyBankCardReqVo();
        updateDefaultReqVo.setIsDefault(CompanyBankCardConstant.IS_NOT_DEFAULT);
        updateDefaultReqVo.setCompanyId(companyBankCardVo.getCompanyId());
        companyBankCardDao.updateDefault(updateDefaultReqVo);

        // 将该卡设置为默认银行卡
        updateDefaultReqVo = new CompanyBankCardReqVo();
        updateDefaultReqVo.setIsDefault(CompanyBankCardConstant.IS_NOT_DEFAULT);
        updateDefaultReqVo.setId(companyBankCardVo.getId());
        companyBankCardDao.updateDefault(updateDefaultReqVo);
    }

}
