package com.pangding.web.authority.mq;

import com.pangding.web.authority.service.CompanyMsgService;
import com.pangding.web.authority.service.CompanyService;
import com.pangding.web.authority.service.FinanceService;
import com.pangding.web.constant.MqConstant;
import com.pangding.web.mq.CompanyProducer;
import com.pangding.web.mq.DistributionProducer;
import com.pangding.web.mq.OrderProducer;
import com.pangding.web.mq.PayProducer;
import com.pangding.web.vo.order.req.CapitalFlowReqVo;
import com.pangding.web.vo.order.req.OrderLogReqVo;
import com.pangding.web.vo.order.req.OrderMoneyDetailReqVo;
import com.pangding.web.vo.order.req.back.CashWithdrawalBackReqVo;
import com.pangding.web.vo.order.tl.CashWithdrawalReqVo;
import com.pangding.web.vo.order.tl.SoundReqVo;
import com.pangding.web.vo.system.req.CompanyMsgReqVo;
import com.pangding.web.vo.system.req.company.UpdateCompanyMoneyReqVo;
import com.pangding.web.vo.system.res.company.CompanyResVo;
import com.rabbitmq.client.Channel;
import com.yanzuoguang.mq.service.MqService;
import com.yanzuoguang.mq.vo.MessageVo;
import com.yanzuoguang.util.helper.DateHelper;
import com.yanzuoguang.util.helper.JsonHelper;
import com.yanzuoguang.util.helper.StringHelper;
import com.yanzuoguang.util.log.Log;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.amqp.core.Message;
import org.springframework.amqp.rabbit.annotation.RabbitListener;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

@Component
public class CompanyConsumer {

    private static final Logger log = LoggerFactory.getLogger(CompanyConsumer.class);

    @Autowired
    private CompanyProducer companyProducer;
    @Autowired
    private PayProducer payProducer;
    @Autowired
    private OrderProducer orderProducer;
    @Autowired
    private DistributionProducer distributionProducer;
    @Autowired
    private MqService mqService;
    @Autowired
    private CompanyMsgService companyMsgService;
    @Autowired
    private FinanceService financeService;
    @Autowired
    private CompanyService companyService;

    /**
     * 初始化mq
     */
    public void init() {
        companyProducer.init();
    }

    /**
     * 保存订单日志mq
     *
     * @param orderLogReqVo
     */
    public void saveOrderLog(OrderLogReqVo orderLogReqVo) {
        orderProducer.saveOrderLog(orderLogReqVo);
    }

    /**
     * 申请提现
     * @param capitalFlowReqVo
     */
    public void applyWithdrawal(CapitalFlowReqVo capitalFlowReqVo){
        companyProducer.applyWithdrawal(capitalFlowReqVo);
    }

//    public void updateCompanyMoneyMQ(UpdateCompanyMoneyReqVo updateCompanyMoneyReqVo){
//        companyProducer.companyMoneyUpdate(updateCompanyMoneyReqVo);
//    }

    @RabbitListener(queues = {MqConstant.COMPANY_MONEY_UPDATE})
    public void updateCompanyMoney(String content, Message message, Channel channel) {
        log.info("[updateCompanyMoney 监听的消息] - [消费时间] - [{}] - [{}]", DateHelper.getNow(), content);
        try {
            UpdateCompanyMoneyReqVo updateCompanyMoneyReqVo = JsonHelper.deserialize(content, UpdateCompanyMoneyReqVo.class);
            financeService.updateCompanyMoney(updateCompanyMoneyReqVo);
        } catch (Exception e) {
            Log.error(CompanyConsumer.class, e);
            mqService.message(new MessageVo(MqConstant.COMPANY_MONEY_UPDATE, MqConstant.COMPANY_MONEY_UPDATE,
                    content, 2000));
        } finally {
            mqService.basicAck(message, channel);
        }
    }

    @RabbitListener(queues = {MqConstant.SAVE_CAPITAL_FLOW})
    public void saveCapitalFlow(String content, Message message, Channel channel) {
        log.info("[saveCapitalFlow 监听的消息] - [消费时间] - [{}] - [{}]", DateHelper.getNow(), content);
        try {
            CapitalFlowReqVo capitalFlowReqVo = JsonHelper.deserialize(content, CapitalFlowReqVo.class);
            financeService.saveCapitalFlow(capitalFlowReqVo);
        } catch (Exception e) {
            Log.error(CompanyConsumer.class, e);
            mqService.message(new MessageVo(MqConstant.SAVE_CAPITAL_FLOW, MqConstant.SAVE_CAPITAL_FLOW,
                    content, 2000));
        } finally {
            mqService.basicAck(message, channel);
        }
    }

    @RabbitListener(queues = {MqConstant.COMPANY_MSG})
    public void saveComapnyMsg(String content, Message message, Channel channel) {
        log.info("[saveComapnyMsg 监听的消息] - [消费时间] - [{}] - [{}]", DateHelper.getNow(), content);
        try {
            CompanyMsgReqVo companyMsgReqVo = JsonHelper.deserialize(content, CompanyMsgReqVo.class);
            companyMsgService.saveCompanyMsg(companyMsgReqVo);
        } catch (Exception e) {
            Log.error(CompanyConsumer.class, e);
        } finally {
            mqService.basicAck(message, channel);
        }
    }


    /**
     * 提现
     *
     * @param cashWithdrawalReqVo
     */
    public void cashWithdrawal(CashWithdrawalReqVo cashWithdrawalReqVo) {
        payProducer.cashWithdrawal(cashWithdrawalReqVo);
    }

    /**
     * 转账
     * @param cashWithdrawalReqVo
     */
    public void divideSettlement(CashWithdrawalReqVo cashWithdrawalReqVo) {
        payProducer.divideSettlement(cashWithdrawalReqVo);
    }

    /**
     * 提现回调
     *
     * @param content
     * @param message
     * @param channel
     */
    @RabbitListener(queues = {MqConstant.CASH_WITHDRAWAL_BACK})
    public void applyMoneyBack(String content, Message message, Channel channel) {
        log.info("[applyMoneyBack 监听的消息] - [消费时间] - [{}] - [{}]", DateHelper.getNow(), content);
        CashWithdrawalBackReqVo req = JsonHelper.deserialize(content, CashWithdrawalBackReqVo.class);
        try {
            financeService.withdrawalBack(req);
            // 保存订单日志MQ
            OrderLogReqVo orderLogReqVo = new OrderLogReqVo(StringHelper.getNewID(), req.getId(), "提现回调",
                    JsonHelper.serialize(req), "", req.getCompanyId(), "");
            saveOrderLog(orderLogReqVo);
        } catch (Exception e) {
            Log.error(CompanyConsumer.class, e);
            // 保存订单日志MQ
            OrderLogReqVo orderLogReqVo = new OrderLogReqVo(StringHelper.getNewID(), req.getId(), "提现回调",
                    JsonHelper.serialize(req), e.getMessage(), req.getCompanyId(), "");
            saveOrderLog(orderLogReqVo);
        } finally {
            mqService.basicAck(message, channel);
        }
    }



    @RabbitListener(queues = {MqConstant.DIVIDE_SETTLEMENT_INIT})
    public void divideSettlementInit(String content, Message message, Channel channel) {
        log.info("[divideSettlementInit 监听的消息] - [消费时间] - [{}] - [{}]", DateHelper.getNow(), content);
        OrderMoneyDetailReqVo req = JsonHelper.deserialize(content, OrderMoneyDetailReqVo.class);
        try {
//            financeService.divideSettlement(req);
            // 保存订单日志MQ
            OrderLogReqVo orderLogReqVo = new OrderLogReqVo(StringHelper.getNewID(), req.getOrderId(), "转账初始化",
                    JsonHelper.serialize(req), "", req.getCompanyId(), "");
//            saveOrderLog(orderLogReqVo);
        } catch (Exception e) {
            Log.error(CompanyConsumer.class, e);
            // 保存订单日志MQ
            OrderLogReqVo orderLogReqVo = new OrderLogReqVo(StringHelper.getNewID(), req.getOrderId(), "转账初始化",
                    JsonHelper.serialize(req), e.getMessage(), req.getCompanyId(), "");
//            saveOrderLog(orderLogReqVo);
        } finally {
            mqService.basicAck(message, channel);
        }
    }

    /**
     * 分润结算回调
     *
     * @param content
     * @param message
     * @param channel
     */
    @RabbitListener(queues = {MqConstant.DIVIDE_SETTLEMENT_BACK})
    public void divideSettlementBack(String content, Message message, Channel channel) {
        log.info("[divideSettlementBack 监听的消息] - [消费时间] - [{}] - [{}]", DateHelper.getNow(), content);
        CashWithdrawalBackReqVo req = JsonHelper.deserialize(content, CashWithdrawalBackReqVo.class);
        try {
            financeService.divideSettlementBack(req);
            // 保存订单日志MQ
            OrderLogReqVo orderLogReqVo = new OrderLogReqVo(StringHelper.getNewID(), req.getId(), "转账回调",
                    JsonHelper.serialize(req), "", req.getCompanyId(), "");
//            saveOrderLog(orderLogReqVo);
        } catch (Exception e) {
            Log.error(CompanyConsumer.class, e);
            // 保存订单日志MQ
            OrderLogReqVo orderLogReqVo = new OrderLogReqVo(StringHelper.getNewID(), req.getId(), "转账回调",
                    JsonHelper.serialize(req), e.getMessage(), req.getCompanyId(), "");
//            saveOrderLog(orderLogReqVo);
        } finally {
            mqService.basicAck(message, channel);
        }
    }

    /**
     *  通联发送语音
     *
     * @param soundReqVo
     */
    public void sendSound(SoundReqVo soundReqVo) {
        payProducer.sendSound(soundReqVo);
    }

    /**
     *  同步公司信息
     *
     * @param companyResVo
     */
    public void synCompany(CompanyResVo companyResVo) {
        distributionProducer.synCompany(companyResVo);
    }

    /**
     * 公司激活
     * @param companyResVo
     */
    public void cmpanyActiviat(CompanyResVo companyResVo) {
        companyProducer.companyActivat(companyResVo);
    }

    @RabbitListener(queues = {MqConstant.COMPANY_ACTIVAT})
    public void activiatCompany(String content, Message message, Channel channel) {
        log.info("[activiatCompany监听的消息] - [消费时间] - [{}] - [{}]", DateHelper.getNow(), content);
        try {
            CompanyResVo companyResVo = JsonHelper.deserialize(content, CompanyResVo.class);
            companyService.activiatCompany(companyResVo);
        } catch (Exception e) {
            Log.error(CompanyConsumer.class, e);
        } finally {
            mqService.basicAck(message, channel);
        }
    }

    /**
     *  公司奖励
     *
     * @param companyResVo
     */
    public void companyReward(CompanyResVo companyResVo) {
        companyProducer.companyReward(companyResVo);
    }

    @RabbitListener(queues = {MqConstant.COMPANY_REWARD})
    public void companyReward(String content, Message message, Channel channel) {
        log.info("[companyReward 监听的消息] - [消费时间] - [{}] - [{}]", DateHelper.getNow(), content);
        try {
            CompanyResVo companyResVo = JsonHelper.deserialize(content, CompanyResVo.class);
            companyService.companyReward(companyResVo);
        } catch (Exception e) {
            Log.error(CompanyConsumer.class, e);
        } finally {
            mqService.basicAck(message, channel);
        }
    }
}
