package com.pangding.web.authority.controller;

import com.pangding.web.authority.service.CompanyService;
import com.pangding.web.authority.vo.CompanyBankCardVo;
import com.pangding.web.authority.vo.CompanyVo;
import com.pangding.web.authority.vo.reqvo.*;
import com.pangding.web.authority.vo.resvo.CompanyResVo;
import com.pangding.web.authority.vo.resvo.DetailResVo;
import com.pangding.web.authority.vo.resvo.SaveCompanyResVo;
import com.yanzuoguang.util.helper.CheckerHelper;
import com.yanzuoguang.util.vo.PageSizeData;
import com.yanzuoguang.util.vo.ResponseResult;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RestController;

/**
 * 公司我的管理controller
 * @author tangfang
 * @date 2019-05-20 10:49
 */
@RestController
@RequestMapping(value="/company")
public class CompanyManagementController {

    @Autowired
    CompanyService companyServiceImpl;

    /**y
     * 新增用户表（密码为123456），公司主表，公司扩展表，商户表，银行卡表（企业类型有，个人类型没有）
     * @param reqVo 公司以及公司扩展表信息
     * @return 公司信息
     */
    @RequestMapping(value = "/saveCompany", method = RequestMethod.POST, produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseResult<SaveCompanyResVo> saveCompany(@RequestBody SaveCompanyReqVo reqVo){
        checkCompany(reqVo);
        return ResponseResult.result(companyServiceImpl.saveCompany(reqVo));
    }

    private void checkCompany(SaveCompanyReqVo reqVo){
        CheckerHelper.newInstance()
                .notBlankCheck("account",reqVo.getAccount())
                .notBlankCheck("companyType",reqVo.getCompanyType())
                .notBlankCheck("companyAttribute",reqVo.getCompanyAttribute())
                .notBlankCheck("merchantName",reqVo.getMerchantName())
                .notBlankCheck("merchantAddress",reqVo.getMerchantAddress())
                .notBlankCheck("companyName",reqVo.getCompanyName())
                .notBlankCheck("companyAddress",reqVo.getCompanyAddress())
                .notBlankCheck("businessLicenseCode",reqVo.getBusinessLicenseCode())
                .notBlankCheck("contactName",reqVo.getContactName())
                .notBlankCheck("contactCode",reqVo.getContactCode())
                .checkIdCard("contactCode",reqVo.getContactCode())
                .notBlankCheck("contactPhone",reqVo.getContactPhone())
                .checkPhoneNo("contactPhone",reqVo.getContactPhone())
                .checkException();
    }


    /**
     * 添加公司提交按钮
     * @param reqVo 封装了公司id和验证码的请求对象
     * @return
     */
    @RequestMapping(value = "/companySubmit", method = RequestMethod.POST, produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseResult companySubmit(@RequestBody SaveCompanyReqVo reqVo){
        checkCompany(reqVo);
        CheckerHelper.newInstance()
                .notBlankCheck("companyId",reqVo.getCompanyId())
                .notBlankCheck("verifyCode",reqVo.getVerifyCode())
                .checkException();
        companyServiceImpl.companySubmit(reqVo);
        return new ResponseResult();
    }

    /**y
     * 商户查看详情，根据公司id返回用户信息和公司主表信息
     * @param reqVo 包含公司id的请求对象
     * @return 封装了用户，公司主表信息的对象
     */
    @RequestMapping(value = "/detail", method = RequestMethod.POST, produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseResult<CompanyResVo> getDetail(@RequestBody CompanyGetReqVo reqVo){
        CheckerHelper.newInstance()
                .notBlankCheck("companyId",reqVo.getId())
                .checkException();
        return ResponseResult.result(companyServiceImpl.getDetail(reqVo));
    }

    /**y
     * 我的二维码页面
     * @param reqVo 公司扩展表请求对象
     * @return 公司主表信息
     */
    @RequestMapping(value = "/myQrCodeUrl", method = RequestMethod.POST, produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseResult<CompanyVo> myQrCodeUrl(@RequestBody CompanyGetReqVo reqVo){
        CheckerHelper.newInstance()
                .notBlankCheck("companyId",reqVo.getId())
                .checkException();
        return ResponseResult.result(companyServiceImpl.myQrCodeUrl(reqVo));
    }

    /**
     * 获取二维码url，有则直接返回，没有则调用微信服务生成url
     * @param reqVo 封装了公司id，页面选项的请求对象
     * @return 二维码url
     */
    @RequestMapping(value = "/generateQrCodeUrl", method = RequestMethod.POST, produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public Object generateQrCodeUrl(@RequestBody GenerateQrCodeUrlReqVo reqVo){
        CheckerHelper.newInstance()
                .notBlankCheck("companyId",reqVo.getCompanyId())
                .notBlankCheck("select",reqVo.getSelect())
                .checkException();
        return companyServiceImpl.generateQrCodeUrl(reqVo);
    }

    /**y
     * 保存/修改银行卡信息，并调用第三方接口发送手机验证码
     *
     * @param reqVo 公司银行卡表信息
     * @return 银行卡表主键
     */
    @RequestMapping(value = "/saveBank", method = RequestMethod.POST, produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseResult<String> saveBank(@RequestBody CompanyBankCardVo reqVo){
        checkBank(reqVo);
        return ResponseResult.result(companyServiceImpl.saveBank(reqVo));
    }

    private void checkBank(CompanyBankCardVo reqVo){
        CheckerHelper.newInstance()
                .notBlankCheck("bankCardCode",reqVo.getBankCardCode())
                .notNumberCheck("bankCardCode",reqVo.getBankCardCode())
                .notBlankCheck("bankName",reqVo.getBankName())
                .notBlankCheck("name",reqVo.getName())
                .notBlankCheck("idCard",reqVo.getIdCard())
                .notBlankCheck("companyId",reqVo.getCompanyId())
                .checkIdCard("idCard",reqVo.getIdCard())
                .notBlankCheck("phone",reqVo.getPhone())
                .checkPhoneNo("phone",reqVo.getPhone())
                .checkException();
    }

    /**y
     * 提交银行卡信息，验证手机验证码，并更新银行卡状态
     * @param reqVo 银行卡表数据+验证码
     * @return 目前返回验证码
     */
    @RequestMapping(value = "/bankSubmit", method = RequestMethod.POST, produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseResult<CompanyBankCardVo> bankSubmit(@RequestBody BankSubmitReqVo reqVo){
        CheckerHelper.newInstance()
                .notBlankCheck("bankCardId",reqVo.getId())
                .notBlankCheck("verifyCode",reqVo.getVerifyCode())
                .checkException();
        return ResponseResult.result(companyServiceImpl.bankSubmit(reqVo));
    }

    /**
     * 获取银行卡列表
     * @param reqVo 继承了分页对象，含公司id
     * @return 银行卡列表分页对象
     */
    @RequestMapping(value = "/bankCardList", method = RequestMethod.POST, produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseResult<PageSizeData<CompanyBankCardVo>> bankCardList(@RequestBody BankCardListReqVo reqVo){
        CheckerHelper.newInstance()
                .notBlankCheck("companyId",reqVo.getCompanyId())
                .checkException();
        return ResponseResult.result(companyServiceImpl.getBankCardList(reqVo));
    }

    /**y
     * 添加用户
     * @param reqVo 封装了用户信息和用户角色信息的请求对象
     * @return 用户id
     */
    @RequestMapping(value = "/saveUser", method = RequestMethod.POST, produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseResult<String> saveUser(@RequestBody WebUserReqVo reqVo) throws Exception {
        CheckerHelper.newInstance()
                .notBlankCheck("account",reqVo.getAccount())
                .notBlankCheck("password",reqVo.getPassword())
                .notBlankCheck("confirmPassword",reqVo.getConfirmPassword())
                .notBlankCheck("name",reqVo.getName())
                .notBlankCheck("email",reqVo.getEmail())
                .notBlankCheck("phone",reqVo.getPhone())
                .checkPhoneNo("phone",reqVo.getPhone())
                .notBlankCheck("createId",reqVo.getCreateId())
                .notBlankCheck("createName",reqVo.getCreateName())
                .notBlankCheck("companyId",reqVo.getCompanyId())
                .checkException();
        return ResponseResult.result(companyServiceImpl.saveUser(reqVo));
    }




    /**
     * 删除银行卡
     * @param reqVo 封装了银行卡id的请求对象
     * @return
     */
    @RequestMapping(value = "/deleteBank", method = RequestMethod.POST, produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseResult deleteBank(@RequestBody CompanyBankCardVo reqVo){
        CheckerHelper.newInstance()
                .notBlankCheck("companyBankCardId",reqVo.getId())
                .checkException();
        companyServiceImpl.deleteBank(reqVo);
        return new ResponseResult();
    }

    /**
     * 设置该银行卡为默认支付，并取消其他银行卡的默认
     * @param reqVo 封装了银行卡id的请求对象
     * @return
     */
    @RequestMapping(value = "/setDefault", method = RequestMethod.POST, produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseResult setDefault(@RequestBody CompanyBankCardVo reqVo){
        CheckerHelper.newInstance()
                .notBlankCheck("bankCardId",reqVo.getId())
                .checkException();
        companyServiceImpl.setDefault(reqVo);
        return new ResponseResult();
    }

    /**
     * 批量删除用户
     * @param reqVo 封装了用户id列表的请求对象
     * @return
     */
    @RequestMapping(value = "/batchDelete", method = RequestMethod.POST, produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseResult batchDelete(@RequestBody BatchDeleteReqVo reqVo){
        CheckerHelper.newInstance()
                .notBlankCheck("userIdList",reqVo.getUserIdList())
                .checkException();
        companyServiceImpl.batchDelete(reqVo);
        return new ResponseResult();
    }

    /**
     * 批量更新用户信息
     * @param reqVo 封装了用户列表的请求对象
     * @return
     */
    @RequestMapping(value = "/batchUpdate", method = RequestMethod.POST, produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseResult batchUpdate(@RequestBody BatchUpdateReqVo reqVo) throws Exception {
        CheckerHelper.newInstance()
                .notBlankCheck("userList",reqVo.getUserVoList())
                .checkException();
        companyServiceImpl.batchUpdate(reqVo);
        return new ResponseResult();
    }

    /**
     * 通过公司id查询该公司邀请的公司
     * @param reqVo 封装了公司id的请求对象
     * @return
     */
    @RequestMapping(value = "/myInvite", method = RequestMethod.POST, produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseResult<PageSizeData<CompanyVo>> myInvite(@RequestBody ListCompanyReqVo reqVo){
        CheckerHelper.newInstance()
                .notBlankCheck("companyId",reqVo.getCompanyId())
                .checkException();
        return ResponseResult.result(companyServiceImpl.myInvite(reqVo));
    }
}
