package com.pangding.web.authority.service.impl;

import com.pangding.web.authority.dao.*;
import com.pangding.web.authority.util.RoleConstant;
import com.pangding.web.authority.vo.reqvo.*;
import com.pangding.web.authority.service.AuthorityService;
import com.pangding.web.authority.vo.resvo.AuthorityListResVo;
import com.pangding.web.authority.vo.resvo.AuthorityResVo;
import com.pangding.web.authority.vo.resvo.WebAuthorityResVo;
import com.pangding.web.vo.system.pd.authority.*;
import com.pangding.web.vo.system.res.authority.AuthorityListByLevelResVo;
import com.yanzuoguang.util.base.ObjectHelper;
import com.yanzuoguang.util.exception.CodeException;
import com.yanzuoguang.util.helper.StringHelper;
import com.yanzuoguang.util.vo.PageSizeData;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;

/**
 * @author zhangjinyao
 */
@Service
public class AuthorityServiceImpl implements AuthorityService {

    @Autowired
    AuthorityDao authorityDao;
    @Autowired
    RoleAuthorityDao roleAuthorityDao;
    @Autowired
    UserDao userDaoImpl;
    @Autowired
    UserRoleDao userRoleDaoImpl;
    @Autowired
    private RegisterServiceImpl registerServiceImpl;
    @Autowired
    private RoleDao roleDaoImpl;

    /**
     * 保存authorityVo对象数据
     *
     * @param authorityVo authorityVo对象
     * @return 用户id
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public AuthorityListResVo save(AuthorityVo authorityVo) {
        /*this.checkValid(authorityVo);*/
        if (null == authorityVo.getPid() || authorityVo.getPid().isEmpty()) {
            authorityVo.setPid("");
        }
        afterPlus(authorityVo);
        authorityVo.setId(StringHelper.getNewID());
        authorityDao.create(authorityVo);

        /*需要给胖丁默认角色绑定上该权限*/
        RoleAuthorityVo roleAuthorityVo = new RoleAuthorityVo();
        roleAuthorityVo.setId(StringHelper.getNewID());
        roleAuthorityVo.setRoleId(RoleConstant.DEFAULT_ROLE_ID);
        roleAuthorityVo.setAuthorityId(authorityVo.getId());
        roleAuthorityDao.create(roleAuthorityVo);

        AuthorityListReqVo reqVo = new AuthorityListReqVo();
        reqVo.setLevelOne("levelOne");
        return this.getListByLevel(reqVo);
    }

    /**
     * 将权限新排序后面所有的权限排序+1
     *
     * @param authorityVo 权限对象
     */
    private void afterPlus(AuthorityVo authorityVo) {
        AuthorityGetReqVo reqVo = new AuthorityGetReqVo();
        reqVo.setPid(authorityVo.getPid());
        reqVo.setSort(authorityVo.getSort());
        reqVo.setAttribute(authorityVo.getAttribute());
        AuthorityVo authorityVo1 = authorityDao.load(reqVo, AuthorityVo.class);
        if (authorityVo1 != null) {
            AuthorityAfterReqVo afterReqVo = new AuthorityAfterReqVo();
            ObjectHelper.writeWithFrom(afterReqVo, authorityVo);
            List<AuthorityVo> authorityVoList = authorityDao.after(afterReqVo);
            for (AuthorityVo authorityVo2 : authorityVoList) {
                Integer sort = authorityVo2.getSort();
                Integer newSort = sort + 1;
                authorityVo2.setSort(newSort);
                authorityDao.update(authorityVo2);
            }
        }
    }

    private void afterSub(AuthorityVo authorityVo) {
        AuthorityGetReqVo getReqVo = new AuthorityGetReqVo();
        getReqVo.setId(authorityVo.getId());
        AuthorityVo authorityVo1 = authorityDao.load(getReqVo, AuthorityVo.class);
        if (authorityVo1 != null) {
            AuthorityAfterReqVo afterReqVo = new AuthorityAfterReqVo();
            ObjectHelper.writeWithFrom(afterReqVo, authorityVo);
            List<AuthorityVo> authorityVoList = authorityDao.after(afterReqVo);
            for (AuthorityVo authorityVo2 : authorityVoList) {
                Integer sort = authorityVo2.getSort();
                Integer newSort = sort - 1;
                authorityVo2.setSort(newSort);
                authorityDao.update(authorityVo2);
            }
        }
    }

    /**
     * 更新authorityVo对象数据
     *
     * @param authorityVo authorityVo对象
     * @return 用户id
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public AuthorityListResVo update(AuthorityVo authorityVo) {
        /*this.checkValid(authorityVo);*/
        if (null == authorityVo.getPid() || authorityVo.getPid().isEmpty()) {
            authorityVo.setPid("");
        }
        updateSort(authorityVo);
        authorityDao.update(authorityVo);
        AuthorityListReqVo reqVo = new AuthorityListReqVo();
        reqVo.setLevelOne("levelOne");
        return this.getListByLevel(reqVo);
    }

    private void updateSort(AuthorityVo authorityVo) {
        AuthorityGetReqVo authorityGetReqVo = new AuthorityGetReqVo();
        authorityGetReqVo.setId(authorityVo.getId());
        AuthorityVo old = authorityDao.load(authorityGetReqVo, AuthorityVo.class);
        if (old.getSort() > authorityVo.getSort()) {
            AuthorityBetweenSortReqVo reqVo = new AuthorityBetweenSortReqVo();
            reqVo.setPid(authorityVo.getPid());
            reqVo.setSmallSort(authorityVo.getSort());
            reqVo.setBigSort(old.getSort());
            reqVo.setAttribute(authorityVo.getAttribute());
            List<AuthorityVo> authorityVoList = authorityDao.getLessBetween(reqVo);
            for (AuthorityVo authorityVo1 : authorityVoList) {
                authorityVo1.setSort(authorityVo1.getSort() + 1);
                authorityDao.update(authorityVo1);
            }
        }
        if (old.getSort() < authorityVo.getSort()) {
            AuthorityBetweenSortReqVo reqVo = new AuthorityBetweenSortReqVo();
            reqVo.setPid(authorityVo.getPid());
            reqVo.setSmallSort(old.getSort());
            reqVo.setBigSort(authorityVo.getSort());
            reqVo.setAttribute(authorityVo.getAttribute());
            List<AuthorityVo> authorityVoList = authorityDao.getMoreBetween(reqVo);
            for (AuthorityVo authorityVo1 : authorityVoList) {
                authorityVo1.setSort(authorityVo1.getSort() - 1);
                authorityDao.update(authorityVo1);
            }
        }
    }

    /**
     * 检查新建、修改的authorityVo对象是否合法
     *
     * @param authorityVo authorityVo对象
     * @return 0：合法 else：不合法
     */
    @Override
    public void checkValid(AuthorityVo authorityVo) {
        authorityDao.checkValid(authorityVo);
    }

    /**
     * 通过权限表主键获取authorityVo对象
     *
     * @param reqVo 主键
     * @return authorityVo对象
     */
    @Override
    public AuthorityResVo getAuthorityById(WebAuthorityReqVo reqVo) {
        AuthorityGetReqVo authorityGetReqVo = new AuthorityGetReqVo();
        authorityGetReqVo.setId(reqVo.getId());
        AuthorityVo authorityVo = authorityDao.load(authorityGetReqVo, AuthorityVo.class);
        if (authorityVo == null) {
            throw new CodeException("该权限不存在");
        }

        AuthorityResVo resVo = new AuthorityResVo();
        ObjectHelper.writeWithFrom(resVo, authorityVo);

        if (null != authorityVo.getPid() && !authorityVo.getPid().isEmpty()) {
            AuthorityGetReqVo getReqVo = new AuthorityGetReqVo();
            getReqVo.setId(authorityVo.getPid());
            AuthorityVo parent = authorityDao.load(getReqVo, AuthorityVo.class);
            if (null != parent) {
                resVo.setParentName(parent.getName());
            }
        }
        return resVo;
    }

    /**
     * 通过权限表主键删除authorityVo对象，并删除权限和角色的对应关系
     * 递归删除其子权限
     *
     * @param reqVo 主键
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteAuthorityById(WebAuthorityReqVo reqVo) {

        /*将该权限排序后的权限排序-1*/
        AuthorityGetReqVo getReqVo = new AuthorityGetReqVo();
        getReqVo.setId(reqVo.getId());
        AuthorityVo authority = authorityDao.load(getReqVo, AuthorityVo.class);
        if (null != authority) {
            afterSub(authority);
        }

        AuthorityGetReqVo reqVo1 = new AuthorityGetReqVo();
        reqVo1.setPid(reqVo.getId());
        List<AuthorityVo> childList = authorityDao.loadList(reqVo1, AuthorityVo.class);
        if (childList != null && !childList.isEmpty()) {
            for (AuthorityVo child : childList) {
                WebAuthorityReqVo reqVo2 = new WebAuthorityReqVo();
                reqVo2.setId(child.getId());
                this.deleteAuthorityById(reqVo2);
            }
        }

        AuthorityVo authorityVo = new AuthorityVo();
        authorityVo.setId(reqVo.getId());
        authorityDao.remove(authorityVo);

        RoleAuthorityGetReqVo roleAuthorityGetReqVo = new RoleAuthorityGetReqVo();
        roleAuthorityGetReqVo.setAuthority(reqVo.getId());
        List<RoleAuthorityVo> roleAuthorityList = roleAuthorityDao.loadList(roleAuthorityGetReqVo, RoleAuthorityVo.class);
        for (RoleAuthorityVo roleAuthority : roleAuthorityList) {
            roleAuthorityDao.remove(roleAuthority);
        }
    }

    /**
     * 获取权限返回信息分页对象
     *
     * @param reqVo
     * @return 权限返回信息分页对象
     */
    @Override
    public PageSizeData<WebAuthorityResVo> getWebAuthorityResVoList(ListAuthorityByPageReqVo reqVo) {
        PageSizeData<AuthorityVo> authorityVoPageSizeData = authorityDao.getAuthorityList(reqVo);
        PageSizeData<WebAuthorityResVo> webAuthorityResVoPageSizeData = new PageSizeData();
        List<AuthorityVo> authorityVoList = authorityVoPageSizeData.getList();
        if (authorityVoList == null || authorityVoList.isEmpty()) {
            throw new CodeException("没有更多权限");
        }
        List<WebAuthorityResVo> webAuthorityResVoList = new ArrayList(authorityVoList.size());
        for (AuthorityVo authorityVo : authorityVoList) {
            WebAuthorityResVo resVo = new WebAuthorityResVo();
            ObjectHelper.writeWithFrom(resVo, authorityVo);
//            if (null != authorityVo.getPid() && !authorityVo.getPid().isEmpty()) {
//                AuthorityGetReqVo authorityGetReqVo = new AuthorityGetReqVo();
//                authorityGetReqVo.setId(authorityVo.getPid());
//                AuthorityVo parent = authorityDao.load(authorityGetReqVo, AuthorityVo.class);
//                if (!StringHelper.isEmpty(parent)) {
//                    resVo.setParentName(parent.getName());
//                } else {
//                    System.out.println("id:" + authorityVo.getPid());
//                }
//
//            }
            webAuthorityResVoList.add(resVo);
        }
        ObjectHelper.writeWithFrom(webAuthorityResVoPageSizeData, authorityVoPageSizeData);
        webAuthorityResVoPageSizeData.setList(webAuthorityResVoList);

        return webAuthorityResVoPageSizeData;
    }

    /**
     * 查询不分页权限列表
     *
     * @return
     */
    @Override
    public List<AuthorityVo> getAuthorityList(AuthorityListReqVo reqVo) {
        List<AuthorityVo> authorityVoList = authorityDao.loadList(reqVo, AuthorityVo.class);
        if (null == authorityVoList || authorityVoList.isEmpty()) {
            throw new CodeException("暂无权限，请先添加");
        }
        return authorityVoList;
    }

    @Override
    public AuthorityListResVo getListByLevel(AuthorityListReqVo reqVo) {
        AuthorityListResVo authorityListResVo = new AuthorityListResVo();
        // 找到该公司管理员角色
        RoleGetReqVo roleGetReqVo = new RoleGetReqVo();
        roleGetReqVo.setCompanyId(reqVo.getCompanyId());
        roleGetReqVo.setIsAdmin(1);
        RoleVo roleVo = roleDaoImpl.load(roleGetReqVo, RoleVo.class);
        if (StringHelper.isEmpty(roleVo)) {
            throw new CodeException("该公司用户未设置管理员角色");
        }

        // 查询该公司管理员角色权限
        RoleAuthorityGetReqVo roleAuthorityGetReqVo = new RoleAuthorityGetReqVo();
        roleAuthorityGetReqVo.setRole(roleVo.getId());
        List<AuthorityListByLevelResVo> roleAuthorityVoList = roleAuthorityDao.loadList(roleAuthorityGetReqVo, AuthorityListByLevelResVo.class);
        authorityListResVo.setAuthorityListByLevel(getChildList(roleAuthorityVoList, false));

        // 查询该角色的权限
        RoleAuthorityGetReqVo roleAuthorityGetReqVo1 = new RoleAuthorityGetReqVo();
        roleAuthorityGetReqVo1.setRole(reqVo.getRoleId());
        List<AuthorityVo> roleAuthorityVoList1 = roleAuthorityDao.loadList(roleAuthorityGetReqVo1, AuthorityVo.class);
        authorityListResVo.setAuthorityListByRole(roleAuthorityVoList1);

        return authorityListResVo;
    }

    public static <T extends AuthorityListByLevelResVo> List<T> getChildList(List<T> list, boolean check) {
        if (list == null || list.isEmpty()) {
            return list;
        }
        Map<String, T> idMap = new HashMap<>(list.size());
        List<T> root = new ArrayList<>();
        // 将所有对象进行映射
        for (T vo : list) {
            String id = vo.getAuthorityId();
            idMap.put(id, vo);
        }

        // 组合成父子级关系
        for (T vo : list) {
            String pid = StringHelper.getFirst(vo.getPid());
            String id = vo.getAuthorityId();
            //当前对象的父对象不存在则创建一个临时的父对象
            if (StringHelper.isEmpty(pid)) {
                root.add(vo);
            } else if (idMap.containsKey(pid)) {
//                List<AuthorityListByLevelResVo> childList = idMap.get(pid).getChildList();
//                if (null == childList) {
//                    childList = new ArrayList<>();
//                }
//                childList.add(vo);
                idMap.get(pid).getChildList().add(vo);
            } else if (check) {
                throw new CodeException(String.format("存在数据不合理,子节点[%s]父id[%s]不存在", id, pid));
            }
        }
        return root;
    }

    private List<AuthorityListByLevelResVo> getChildList(String id) {
        AuthorityGetReqVo reqVo = new AuthorityGetReqVo();
        reqVo.setPid(id);
        List<AuthorityVo> childList = authorityDao.listBySort(reqVo);
        List<AuthorityListByLevelResVo> childResList = new ArrayList();
        if (null != childList && !childList.isEmpty()) {
            for (AuthorityVo authorityVo : childList) {
                AuthorityListByLevelResVo resVo = new AuthorityListByLevelResVo();
                ObjectHelper.writeWithFrom(resVo, authorityVo);
                List<AuthorityListByLevelResVo> grandchildList = getChildList(authorityVo.getId());
                resVo.setChildList(grandchildList);
                childResList.add(resVo);
            }
        }
        return childResList;
    }

    @Override
    public List<AuthorityListByLevelResVo> getCompanyAuthority(CompanyAuthorityReqVo reqVo) {
        UserGetReqVo userGetReqVo = new UserGetReqVo();
        userGetReqVo.setCompanyId(reqVo.getCompanyId());
        userGetReqVo.setIsFirst(1);
        UserVo superAdmin = userDaoImpl.load(userGetReqVo, UserVo.class);
        if (null == superAdmin) {
            throw new CodeException("暂无超级管理员");
        }
        UserRoleGetReqVo getReqVo = new UserRoleGetReqVo();
        getReqVo.setUserId(superAdmin.getId());
        List<UserRoleVo> userRoleVoList = userRoleDaoImpl.loadList(getReqVo, UserRoleVo.class);
        if (null == userRoleVoList || userRoleVoList.isEmpty()) {
            throw new CodeException("暂无权限");
        }
        List<RoleAuthorityVo> roleAuthorityVoList = new ArrayList();
        for (UserRoleVo userRoleVo : userRoleVoList) {
            RoleAuthorityGetReqVo roleAuthorityGetReqVo = new RoleAuthorityGetReqVo();
            roleAuthorityGetReqVo.setRoleId(userRoleVo.getRoleId());
            List<RoleAuthorityVo> roleAuthorityVos = roleAuthorityDao.loadList(roleAuthorityGetReqVo, RoleAuthorityVo.class);
            roleAuthorityVoList.addAll(roleAuthorityVos);
        }
        if (null == roleAuthorityVoList || roleAuthorityVoList.isEmpty()) {
            throw new CodeException("暂无权限");
        }
        List<AuthorityVo> authorityVoList = new ArrayList();
        for (RoleAuthorityVo roleAuthorityVo : roleAuthorityVoList) {
            AuthorityGetReqVo authorityGetReqVo = new AuthorityGetReqVo();
            authorityGetReqVo.setId(roleAuthorityVo.getAuthorityId());
            AuthorityVo authorityVo = authorityDao.load(authorityGetReqVo, AuthorityVo.class);
            if (authorityVo != null) {
                authorityVoList.add(authorityVo);
            }
        }
        if (null == authorityVoList || authorityVoList.isEmpty()) {
            throw new CodeException("暂无权限");
        }
        List<AuthorityListByLevelResVo> resVoList = new ArrayList();
        for (AuthorityVo authorityVo : authorityVoList) {
            AuthorityListByLevelResVo resVo = new AuthorityListByLevelResVo();
            ObjectHelper.writeWithFrom(resVo, authorityVo);
            List<AuthorityListByLevelResVo> childList = getChildList(authorityVo.getId());
            resVo.setChildList(childList);
            resVoList.add(resVo);
        }
        return resVoList;
    }
}
