package com.pangding.web.authority.service.impl;

import com.pangding.web.authority.dao.CompanyBankCardDao;
import com.pangding.web.authority.dao.CompanyDao;
import com.pangding.web.authority.dao.CompanyExtendDao;
import com.pangding.web.authority.dao.UserDao;
import com.pangding.web.authority.feign.PayFeign;
import com.pangding.web.authority.feign.StoreFeign;
import com.pangding.web.authority.service.CompanyService;
import com.pangding.web.authority.service.UserService;
import com.pangding.web.authority.util.CompanyBankCardConstant;
import com.pangding.web.authority.util.CompanyConstant;
import com.pangding.web.authority.util.RsaConstant;
import com.pangding.web.authority.vo.*;
import com.pangding.web.authority.vo.reqvo.*;
import com.pangding.web.authority.vo.resvo.CompanyResVo;
import com.pangding.web.authority.vo.resvo.DetailResVo;
import com.pangding.web.authority.vo.resvo.SaveCompanyResVo;
import com.pangding.web.util.RSAUtils;
import com.yanzuoguang.util.base.ObjectHelper;
import com.yanzuoguang.util.exception.CodeException;
import com.yanzuoguang.util.helper.DateHelper;
import com.yanzuoguang.util.helper.StringHelper;
import com.yanzuoguang.util.vo.PageSizeData;
import com.yanzuoguang.util.vo.ResponseResult;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.List;

/**
 * @author Administrator
 */
@Service
public class CompanyServiceImpl implements CompanyService{

    @Autowired
    private CompanyDao companyDao;
    @Autowired
    private CompanyExtendDao companyExtendDaoImpl;
    @Autowired
    private CompanyBankCardDao companyBankCardDao;
    @Autowired
    private UserService userServiceImpl;
    @Autowired
    private UserDao userDaoImpl;
    @Autowired
    PayFeign payFeign;
    @Autowired
    StoreFeign storeFeign;

    /**
     * 修改公司信息
     * @param companyVo
     */
    @Override
    public void updateCompanyStatus(CompanyVo companyVo){
        CompanyVo company = companyDao.load(companyVo.getId(), CompanyVo.class);
        ObjectHelper.writeWithFrom(company, companyVo);
        companyDao.update(company);
    }

    /**
     * 分页查询公司信息
     * @param reqVo 继承了分页对象
     * @return 公司列表分页对象
     */
    @Override
    public PageSizeData<CompanyVo> list(ListCompanyReqVo reqVo) {
        PageSizeData<CompanyVo> pageSizeData = companyDao.getCompanyList(reqVo);
        if (null == pageSizeData || null == pageSizeData.getList() || pageSizeData.getList().isEmpty()){
            throw new CodeException("暂无公司信息");
        }
        return pageSizeData;
    }

    /**
     * 胖丁查询用户，公司和公司扩展表信息以及银行卡列表
     * @param reqVo 包含公司id的请求对象
     * @return 封装了用户，公司，公司扩展表和银行卡列表信息的对象
     */
    @Override
    public CompanyResVo pdGetDetail(CompanyGetReqVo reqVo) {
        CompanyResVo resVo = new CompanyResVo();
        /*获取公司表信息*/
        CompanyVo companyVo = companyDao.load(reqVo,CompanyVo.class);
        if (null != companyVo){
            resVo.setCompanyVo(companyVo);
        }

        /*此处需要根据公司id和productOrInterface获取商户信息*/
        CompanyScenicInfoReqVo companyScenicInfoReqVo = new CompanyScenicInfoReqVo();
        companyScenicInfoReqVo.setCompanyId(companyVo.getId());
        companyScenicInfoReqVo.setProductOrInterface(companyVo.getProductOrInterface());
        companyScenicInfoReqVo.setReqId(reqVo.getReqId());
        try {
            ResponseResult result = storeFeign.getCompanyScenicInfo(companyScenicInfoReqVo);
            if (null != result && "00".equals(result.getCode())){
                resVo.setMerchant((Merchant) result.getData());
            }
        }catch (Exception e){
            resVo.setMerchant(null);
        }

        /*获取公司扩展表信息*/
        CompanyExtendGetReqVo companyExtendGetReqVo = new CompanyExtendGetReqVo();
        companyExtendGetReqVo.setId(reqVo.getId());
        CompanyExtendVo companyExtendVo = companyExtendDaoImpl.load(companyExtendGetReqVo,CompanyExtendVo.class);
        if (null != companyExtendVo){
            resVo.setCompanyExtendVo(companyExtendVo);
        }
        /*获取公司银行卡列表*/
        CompanyBankCardGetReqVo companyBankCardGetReqVo = new CompanyBankCardGetReqVo();
        companyBankCardGetReqVo.setCompanyId(reqVo.getId());
        List<CompanyBankCardVo> companyBankCardVoList = companyBankCardDao.loadList(companyBankCardGetReqVo,CompanyBankCardVo.class);
        if (null != companyBankCardVoList && !companyBankCardVoList.isEmpty()){
            resVo.setBankCardVoList(companyBankCardVoList);
        }
        /*获取用户表信息*/
        UserGetReqVo userGetReqVo = new UserGetReqVo();
        userGetReqVo.setCompanyId(reqVo.getId());
        List<UserVo> userVoList = userDaoImpl.loadList(userGetReqVo,UserVo.class);
        if (null != userVoList){
            for (UserVo userVo:userVoList) {
                userVo.setPassword("");
                userVo.setRemark("");
            }
        }
        resVo.setUserVoList(userVoList);
        return resVo;
    }

    /**
     * 商户查看详情，根据公司id返回用户信息和公司主表信息
     * @param reqVo 包含公司id的请求对象
     * @return 封装了用户，公司主表信息的对象
     */
    @Override
    public DetailResVo getDetail(CompanyGetReqVo reqVo) {
        DetailResVo resVo = new DetailResVo();
        /*查询公司主表信息*/
        CompanyVo companyVo = companyDao.load(reqVo,CompanyVo.class);
        if (null != companyVo){
            resVo.setCompanyVo(companyVo);
        }
        /*查询用户信息*/
        UserGetReqVo userGetReqVo = new UserGetReqVo();
        userGetReqVo.setCompanyId(reqVo.getId());
        List<UserVo> userVoList = userDaoImpl.loadList(userGetReqVo,UserVo.class);
        if (null != userVoList){
            for (UserVo userVo:userVoList) {
                userVo.setPassword("");
                userVo.setRemark("");
            }
        }
        resVo.setUserVoList(userVoList);
        return resVo;
    }

    /**
     * 新增用户表（密码为123456），公司主表，公司扩展表，商户表，银行卡表（企业类型有，个人类型没有）
     * @param reqVo 公司以及公司扩展表信息
     * @return 公司id
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public SaveCompanyResVo saveCompany(SaveCompanyReqVo reqVo) {
        /*返回结果对象*/
        SaveCompanyResVo resVo = new SaveCompanyResVo();
        /*公司主表*/
        CompanyVo companyVo = new CompanyVo();
        ObjectHelper.writeWithFrom(companyVo,reqVo);
        companyVo.setId(StringHelper.getNewID());
        companyVo.setAddress(reqVo.getCompanyAddress());
        companyVo.setStatus(0);

        /*调用支付服务注册*/
        CompanyReqVo companyReqVo = new CompanyReqVo();
        ObjectHelper.writeWithFrom(companyReqVo,companyVo);

        if ("1".equals(reqVo.getCompanyType())){
            /*银行卡表*/
            CompanyBankCardVo bankCardVo = new CompanyBankCardVo();
            ObjectHelper.writeWithFrom(bankCardVo,reqVo);
            bankCardVo.setId(StringHelper.getNewID());
            bankCardVo.setCompanyId(companyVo.getId());
            bankCardVo.setName(reqVo.getContactName());
            bankCardVo.setPhone(reqVo.getContactPhone());
            bankCardVo.setIdCard(reqVo.getContactCode());
            bankCardVo.setType(reqVo.getCompanyType());
            bankCardVo.setStatus(0);

            CompanyBankReqVo companyBankReqVo = new CompanyBankReqVo();
            ObjectHelper.writeWithFrom(companyBankReqVo,bankCardVo);
            companyReqVo.getBankList().add(companyBankReqVo);
        }

        /*注册*/
        ResponseResult result = payFeign.setCompanyInfo(companyReqVo);
        if (null == result || !"00".equals(result.getCode())){
            throw new CodeException("保存失败，请重试");
        }
        CompanyReqVo res = (CompanyReqVo) result.getData();
        CompanyVo companyVo1 = new CompanyVo();
        ObjectHelper.writeWithFrom(companyVo1,res);
        companyVo1.setProductOrInterface(CompanyConstant.PRODUCT);

        /*保存公司主表*/
        companyDao.save(companyVo1);
        resVo.setCompanyId(companyVo1.getId());
        /*保存银行卡表*/
        List<CompanyBankReqVo> companyBankReqVoList = res.getBankList();
        if (null != companyBankReqVoList || !companyBankReqVoList.isEmpty()){
            CompanyBankCardVo companyBankCardVo = new CompanyBankCardVo();
            CompanyBankReqVo companyBankReqVo = companyBankReqVoList.get(0);
            ObjectHelper.writeWithFrom(companyBankCardVo,companyBankReqVo);
            companyBankCardVo.setIsDefault(CompanyBankCardConstant.IS_DEFAULT);
            companyBankCardDao.save(companyBankCardVo);
            resVo.setCompanyBankCardId(companyBankCardVo.getId());
        }

        /*保存公司扩展表*/
        CompanyExtendVo extendVo = new CompanyExtendVo();
        extendVo.setId(companyVo.getId());
        String qrCode = DateHelper.getNow()
                .replace("-","")
                .replace(":","")
                .replace(" ","");
        extendVo.setQrCode(qrCode);
        companyExtendDaoImpl.save(extendVo);
        resVo.setCompanyExtendId(extendVo.getId());

        /*保存商户表*/
        MerchantCreateReqVo merchantCreateReqVo = new MerchantCreateReqVo();
        merchantCreateReqVo.setId(StringHelper.getNewID());
        merchantCreateReqVo.setType(reqVo.getCompanyAttribute());
        merchantCreateReqVo.setName(reqVo.getMerchantName());
        merchantCreateReqVo.setAddress(reqVo.getMerchantAddress());
        merchantCreateReqVo.setUserId(reqVo.getCreateId());
        merchantCreateReqVo.setUsername(reqVo.getCreateName());
        merchantCreateReqVo.setProductCompanyId(companyVo.getId());
        ResponseResult result1 = storeFeign.merchantCreate(merchantCreateReqVo);
        if (null == result1 || !"00".equals(result1.getCode())){
            throw new CodeException("保存失败，请重试");
        }
        String merchantId = (String) result1.getData();
        resVo.setMerchantId(merchantId);

        /*保存用户*/
        UserVo userVo = new UserVo();
        userVo.setId(StringHelper.getNewID());
        userVo.setAccount(reqVo.getAccount());
        userVo.setPassword(StringHelper.md5("123456"));
        userVo.setRemark("123456");
        userVo.setCreateId(reqVo.getCreateId());
        userVo.setCreateName(reqVo.getCreateName());
        userVo.setCompanyId(companyVo.getId());
        userDaoImpl.save(userVo);
        resVo.setUserId(userVo.getId());

        return resVo;
    }

    /**
     * 添加公司提交按钮
     * @param reqVo 封装了公司id和验证码的请求对象
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void companySubmit(CompanySubmitReqVo reqVo) {
        CompanyGetReqVo companyGetReqVo = new CompanyGetReqVo();
        companyGetReqVo.setId(reqVo.getCompanyId());
        CompanyVo companyVo = companyDao.load(companyGetReqVo,CompanyVo.class);
        CompanyReqVo companyReqVo = new CompanyReqVo();
        ObjectHelper.writeWithFrom(companyReqVo,companyVo);
        companyReqVo.setValidateCode(reqVo.getVerifyCode());
        /*若公司类型为企业，则需要加入银行卡*/
        if (companyVo.getCompanyType() == 1){
            CompanyBankCardGetReqVo companyBankCardGetReqVo = new CompanyBankCardGetReqVo();
            companyBankCardGetReqVo.setCompanyId(reqVo.getCompanyId());
            List<CompanyBankCardVo> companyBankCardVoList = companyBankCardDao.loadList(companyBankCardGetReqVo,CompanyBankCardVo.class);
            List<CompanyBankReqVo> companyBankReqVoList = new ArrayList<>();
            if (null != companyBankCardVoList && !companyBankCardVoList.isEmpty()){
                CompanyBankReqVo companyBankReqVo = new CompanyBankReqVo();
                ObjectHelper.writeWithFrom(companyBankCardGetReqVo,companyBankReqVoList.get(0));
                companyBankReqVoList.add(companyBankReqVo);
            }
            companyReqVo.setBankList(companyBankReqVoList);
        }
        /*调用支付接口验证验证码*/
        ResponseResult result = payFeign.setCompanyInfo(companyReqVo);
        if (null == result || !"00".equals(result.getCode())){
            throw new CodeException("保存出错，请重试");
        }
        CompanyReqVo companyReqVo1 = (CompanyReqVo) result.getData();
        CompanyVo companyVo1 = new CompanyVo();
        ObjectHelper.writeWithFrom(companyVo1,companyReqVo1);
        companyVo1.setProductOrInterface(companyVo.getProductOrInterface());
        companyDao.update(companyVo1);
        /*若公司类型为企业，则需要修改银行卡*/
        if (companyVo1.getCompanyType() == 1){
            CompanyBankReqVo companyBankReqVo = companyReqVo1.getBankList().get(0);
            CompanyBankCardVo companyBankCardVo = new CompanyBankCardVo();
            ObjectHelper.writeWithFrom(companyBankCardVo,companyBankReqVo);
            companyBankCardVo.setIsDefault(CompanyBankCardConstant.IS_DEFAULT);
            companyBankCardDao.update(companyBankCardVo);
        }
    }

    /**
     * 仅将公司状态设置为12（停用）
     * @param reqVo 封装了公司id 的请求对象
     * @return
     */
    @Override
    public void deleteCompany(DeleteCompanyReqVo reqVo) {
        /*仅修改公司状态status，不删除数据*/
        CompanyGetReqVo getReqVo = new CompanyGetReqVo();
        getReqVo.setId(reqVo.getCompanyId());
        CompanyVo companyVo = companyDao.load(getReqVo,CompanyVo.class);
        companyVo.setStatus(CompanyConstant.COMPANY_STATUS_STOP);
        companyVo.setUpdateTime(DateHelper.getNow());
        companyDao.update(companyVo);
    }

    /**
     * 保存/更新银行卡信息，并调用第三方接口发送手机验证码
     *
     * @param reqVo 公司银行卡表信息
     * @return 银行卡表主键
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public String saveBank(CompanyBankCardVo reqVo) {
        String id = StringHelper.getNewID();

        /*若该卡存在，则修改*/
        CompanyBankCardGetReqVo getReqVo = new CompanyBankCardGetReqVo();
        if (null != reqVo.getId() && !"".equals(reqVo.getId())){
            getReqVo.setId(reqVo.getId());
            CompanyBankCardVo companyBankCardVo = companyBankCardDao.load(getReqVo,CompanyBankCardVo.class);
            if (null != companyBankCardVo){
                companyBankCardDao.update(reqVo);
            }
        }
        /*若卡不存在，则新增*/
        else{
            reqVo.setId(id);
            BankCardListReqVo reqVo1 = new BankCardListReqVo();
            PageSizeData<CompanyBankCardVo> companyBankCardDaoBankCardList = companyBankCardDao.getBankCardList(reqVo1);

            /*若该银行卡是第一张银行卡，则设为默认*/
            if (null == companyBankCardDaoBankCardList || companyBankCardDaoBankCardList.getList().isEmpty()){
                reqVo.setIsDefault(CompanyBankCardConstant.IS_DEFAULT);
            }
            companyBankCardDao.create(reqVo);
        }

        /*此处需要调用第三方接口发送手机验证码，并修改数据库状态*/
        CompanyBankReqVo companyBankReqVo = new CompanyBankReqVo();
        ObjectHelper.writeWithFrom(companyBankReqVo,reqVo);
        ResponseResult result = payFeign.bindBack(companyBankReqVo);
        if (null == result || !"00".equals(result.getCode())){
            throw new CodeException("绑定银行卡失败，请重试");
        }
        CompanyBankReqVo resVo = (CompanyBankReqVo) result.getData();
        CompanyBankCardVo companyBankCardVo = new CompanyBankCardVo();
        ObjectHelper.writeWithFrom(companyBankCardVo,resVo);
        companyBankCardVo.setIsDefault(reqVo.getIsDefault());
        companyBankCardDao.update(companyBankCardVo);
        return reqVo.getId();
    }

    /**
     * 调用支付服务解绑银行卡，并更新银行卡状态
     * @param reqVo 封装了银行卡id的请求对象
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteBank(CompanyBankCardVo reqVo) {
        CompanyBankCardVo bankCardVo = companyBankCardDao.load(reqVo,CompanyBankCardVo.class);
        CompanyBankReqVo companyBankReqVo = new CompanyBankReqVo();
        ObjectHelper.writeWithFrom(companyBankReqVo,bankCardVo);
        ResponseResult result = payFeign.unbindBankCard(companyBankReqVo);
        if (null == result || !"00".equals(result.getCode())){
            throw new CodeException("解绑失败，请重试");
        }
        companyBankCardDao.remove(bankCardVo);
    }

    /**
     * 设置该银行卡为默认支付，并取消其他银行卡的默认
     * @param reqVo 封装了银行卡id的请求对象
     * @return
     */
    @Override
    public void setDefault(CompanyBankCardVo reqVo) {
        /*获取该银行卡的公司id*/
        CompanyBankCardVo companyBankCardVo = companyBankCardDao.load(reqVo,CompanyBankCardVo.class);
        String companyId = companyBankCardVo.getCompanyId();
        /*现将该公司其他银行卡设为非默认*/
        CompanyBankCardSetNotDefaultReqVo setNotDefaultReqVo = new CompanyBankCardSetNotDefaultReqVo();
        setNotDefaultReqVo.setCompanyId(companyId);
        companyBankCardDao.setNotDefault(setNotDefaultReqVo);
        /*将该卡设置为默认银行卡并更新*/
        companyBankCardVo.setIsDefault(CompanyBankCardConstant.IS_DEFAULT);
        companyBankCardDao.update(companyBankCardVo);
    }

    /**
     * 提交银行卡信息，验证手机验证码，并更新银行卡状态
     * @param reqVo 银行卡表数据+验证码
     * @return 目前返回验证码
     */
    @Override
    public CompanyBankCardVo bankSubmit(BankSubmitReqVo reqVo) {
        /*根据银行卡id获取银行卡信息*/
        CompanyBankCardGetReqVo getReqVo = new CompanyBankCardGetReqVo();
        getReqVo.setId(reqVo.getId());
        CompanyBankCardVo companyBankCardVo = companyBankCardDao.load(getReqVo,CompanyBankCardVo.class);

        /*此处应该将companyBankCardVo和verifyCode发送到第三方接口，获取到的返回值用于更新银行卡状态*/
        CompanyBankReqVo companyBankReqVo = new CompanyBankReqVo();
        ObjectHelper.writeWithFrom(companyBankReqVo,companyBankCardVo);
        companyBankReqVo.setValidateCode(reqVo.getVerifyCode());
        ResponseResult result = payFeign.bindBack(companyBankReqVo);
        if (null == result || !"00".equals(result.getCode())){
            throw new CodeException("绑定银行卡失败，请重试");
        }
        CompanyBankReqVo resVo = (CompanyBankReqVo) result.getData();
        CompanyBankCardVo companyBankCardVo1 = new CompanyBankCardVo();
        ObjectHelper.writeWithFrom(companyBankCardVo1,resVo);
        companyBankCardVo1.setIsDefault(companyBankCardVo.getIsDefault());
        companyBankCardDao.update(companyBankCardVo1);

        return companyBankCardVo1;
    }

    /**y
     * 我的二维码页面
     * @param reqVo 公司扩展表请求对象
     * @return 公司主表信息
     */
    @Override
    public CompanyVo myQrCodeUrl(CompanyGetReqVo reqVo) {
        CompanyVo companyVo = companyDao.load(reqVo,CompanyVo.class);
        if (null == companyVo){
            throw new CodeException("暂无公司信息");
        }
        return companyVo;
    }

    /**
     * 获取二维码url，有则直接返回，没有则调用微信服务生成url
     * @param reqVo 封装了公司id，页面选项的请求对象
     * @return 二维码url
     */
    @Override
    public String generateQrCodeUrl(GenerateQrCodeUrlReqVo reqVo) {
        CompanyExtendGetReqVo reqVo1 = new CompanyExtendGetReqVo();
        reqVo1.setId(reqVo.getCompanyId());
        CompanyExtendVo companyExtendVo = companyExtendDaoImpl.load(reqVo1,CompanyExtendVo.class);
        if (null == companyExtendVo){
            throw new CodeException("暂无公司扩展表信息");
        }
        String url = companyExtendVo.getQrCodeUrl();
        if (null != url && !url.isEmpty()){
            return url;
        }
        String qrCode = companyExtendVo.getQrCode();
        if (null == qrCode || qrCode.isEmpty()){
            qrCode = DateHelper.getNow()
                    .replace("-","")
                    .replace(":","")
                    .replace(" ","");
        }
        int select = reqVo.getSelect();
        /*此处应调用微信接口传入qrCode和select获取qrCodeUrl*/
        return null;
    }

    /**
     * 获取银行卡列表
     * @param reqVo 继承了分页对象，含公司id
     * @return 银行卡列表分页对象
     */
    @Override
    public PageSizeData<CompanyBankCardVo> getBankCardList(BankCardListReqVo reqVo) {
        PageSizeData<CompanyBankCardVo> companyBankCardVoPageSizeData = companyBankCardDao.getBankCardList(reqVo);
        if (companyBankCardVoPageSizeData == null || companyBankCardVoPageSizeData.getList().isEmpty()) {
            throw new CodeException("暂无银行卡，请先添加");
        }
        return companyBankCardVoPageSizeData;
    }

    /**
     * 添加用户
     * @param reqVo 封装了用户信息和用户角色信息的请求对象
     * @return 用户id
     */
    @Override
    public String saveUser(WebUserReqVo reqVo) throws Exception {
        return userServiceImpl.saveUser(reqVo);
    }

    /**
     * 根据请求的数据返回对应的供应商或是接口商列表
     * @param reqVo
     * @return
     */
    @Override
    public List<CompanyVo> productOrInterface(ProductOrInterfaceReqVo reqVo) {
        CompanyGetReqVo getReqVo = new CompanyGetReqVo();
        ObjectHelper.writeWithFrom(getReqVo,reqVo);
        List<CompanyVo> companyVoList = companyDao.loadList(getReqVo,CompanyVo.class);
        if (null == companyVoList || companyVoList.isEmpty()){
            throw new CodeException("未找到公司信息");
        }
        return companyVoList;
    }

    /**
     * 批量删除用户
     * @param reqVo 封装了用户id列表的请求对象
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void batchDelete(BatchDeleteReqVo reqVo) {
        List<String> userIdList = reqVo.getUserIdList();
        if (null == userIdList || userIdList.isEmpty()){
            throw new CodeException("删除列表为空，请添加");
        }
        for (String userId:userIdList) {
            UserGetReqVo getReqVo = new UserGetReqVo();
            getReqVo.setId(userId);
            userDaoImpl.remove(getReqVo);
        }
    }

    /**
     * 批量更新用户信息
     * @param reqVo 封装了用户列表的请求对象
     * @return
     */
    @Override
    public void batchUpdate(BatchUpdateReqVo reqVo) throws Exception {
        List<UserVo> userVoList = reqVo.getUserVoList();
        if (null == userVoList || userVoList.isEmpty()){
            throw new CodeException("更新列表为空，请添加");
        }
        for (UserVo userVo:userVoList) {
            UserGetReqVo getReqVo = new UserGetReqVo();
            getReqVo.setId(userVo.getId());
            UserVo userVoOld = userDaoImpl.load(getReqVo,UserVo.class);
            if (!StringHelper.compare(userVo.getAccount(),userVoOld.getAccount())){
                userServiceImpl.isValid(userVo.getAccount());
                userServiceImpl.checkAccount(userVo);
            }
            if ("".equals(userVo.getPassword())){
                userVo.setPassword(userVoOld.getPassword());
                userVo.setRemark(userVoOld.getRemark());
            }else{
                String password = RSAUtils.decryptionByPrivateKey(userVo.getPassword(), RsaConstant.privateKey);
                userVo.setPassword(StringHelper.md5(password));
                userVo.setRemark(password);
            }
            if (!StringHelper.compare(userVo.getEmail(),userVoOld.getEmail())){
                userServiceImpl.checkEmail(userVo.getEmail());
            }
            if (!StringHelper.compare(userVo.getPhone(),userVoOld.getPhone())){
                userServiceImpl.checkPhone(userVo);
            }
            userDaoImpl.update(userVo);
        }
    }
}
