package com.pangding.web.authority.service.impl;

import com.pangding.web.authority.dao.RoleAuthorityDao;
import com.pangding.web.authority.dao.AuthorityDao;
import com.pangding.web.authority.vo.reqvo.*;
import com.pangding.web.authority.service.AuthorityService;
import com.pangding.web.authority.vo.*;
import com.yanzuoguang.util.base.ObjectHelper;
import com.yanzuoguang.util.exception.CodeException;
import com.yanzuoguang.util.helper.StringHelper;
import com.yanzuoguang.util.vo.PageSizeData;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.List;

/**
 * @author zhangjinyao
 */
@Service
public class AuthorityServiceImpl implements AuthorityService {

    @Autowired
    AuthorityDao authorityDao;

    @Autowired
    RoleAuthorityDao roleAuthorityDao;

    /**
     * 保存authorityVo对象数据
     *
     * @param authorityVo authorityVo对象
     * @return 用户id
     */
    @Override
    public String save(AuthorityVo authorityVo) {
        this.checkValid(authorityVo);
        authorityVo.setId(StringHelper.getNewID());
        authorityDao.create(authorityVo);
        return authorityVo.getId();
    }

    /**
     * 更新authorityVo对象数据
     *
     * @param authorityVo authorityVo对象
     * @return 用户id
     */
    @Override
    public String update(AuthorityVo authorityVo) {
        this.checkValid(authorityVo);
        authorityDao.update(authorityVo);
        return authorityVo.getId();
    }

    /**
     * 检查新建、修改的authorityVo对象是否合法
     *
     * @param authorityVo authorityVo对象
     * @return 0：合法 else：不合法
     */
    @Override
    public void checkValid(AuthorityVo authorityVo) {
        authorityDao.checkValid(authorityVo);
    }

    /**
     * 通过权限表主键获取authorityVo对象
     *
     * @param reqVo 主键
     * @return authorityVo对象
     */
    @Override
    public AuthorityVo getAuthorityById(WebAuthorityReqVo reqVo) {
        AuthorityGetReqVo authorityGetReqVo = new AuthorityGetReqVo();
        authorityGetReqVo.setId(reqVo.getId());
        AuthorityVo authorityVo = authorityDao.load(authorityGetReqVo,AuthorityVo.class);
        if (authorityVo == null){
            throw new CodeException("该权限不存在");
        }
        return authorityVo;
    }

    /**
     * 通过权限表主键删除authorityVo对象，并删除权限和角色的对应关系
     * 递归删除其子权限
     *
     * @param reqVo 主键
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteAuthorityById(WebAuthorityReqVo reqVo) {

        AuthorityGetReqVo reqVo1 = new AuthorityGetReqVo();
        reqVo1.setPid(reqVo.getId());
        List<AuthorityVo> childList = authorityDao.loadList(reqVo1,AuthorityVo.class);
        if (childList != null && !childList.isEmpty()){
            for (AuthorityVo child:childList) {
                WebAuthorityReqVo reqVo2 = new WebAuthorityReqVo();
                reqVo2.setId(child.getId());
                this.deleteAuthorityById(reqVo2);
            }
        }

        AuthorityVo authorityVo = new AuthorityVo();
        authorityVo.setId(reqVo.getId());
        authorityDao.remove(authorityVo);

        RoleAuthorityGetReqVo roleAuthorityGetReqVo = new RoleAuthorityGetReqVo();
        roleAuthorityGetReqVo.setAuthorityId(reqVo.getId());
        List<RoleAuthorityVo> roleAuthorityList = roleAuthorityDao.loadList(roleAuthorityGetReqVo,RoleAuthorityVo.class);
        for (RoleAuthorityVo roleAuthority:roleAuthorityList) {
            roleAuthorityDao.remove(roleAuthority);
        }
    }

    /**
     * 获取权限返回信息分页对象
     *
     * @param reqVo
     * @return 权限返回信息分页对象
     */
    @Override
    public PageSizeData<WebAuthorityResVo> getWebAuthorityResVoList(ListAuthorityReqVo reqVo) {
        PageSizeData<AuthorityVo> authorityVoPageSizeData = authorityDao.getAuthorityList(reqVo);
        PageSizeData<WebAuthorityResVo> webAuthorityResVoPageSizeData = new PageSizeData();
        List<AuthorityVo> authorityVoList = authorityVoPageSizeData.getList();
        if (authorityVoList == null || authorityVoList.isEmpty()){
            throw new CodeException("没有更多权限");
        }
        List<WebAuthorityResVo> webAuthorityResVoList = new ArrayList(authorityVoList.size());
        for (AuthorityVo authorityVo:authorityVoList) {
            WebAuthorityResVo resVo = new WebAuthorityResVo();
            ObjectHelper.writeWithFrom(resVo,authorityVo);
            webAuthorityResVoList.add(resVo);
        }
        ObjectHelper.writeWithFrom(webAuthorityResVoPageSizeData,authorityVoPageSizeData);
        webAuthorityResVoPageSizeData.setList(webAuthorityResVoList);

        return webAuthorityResVoPageSizeData;
    }
}
