package com.pangding.web.authority.service.impl;

import com.pangding.web.authority.dao.*;
import com.pangding.web.authority.vo.reqvo.*;
import com.pangding.web.authority.service.RoleService;
import com.pangding.web.authority.vo.resvo.WebRoleResVo;
import com.pangding.web.constant.CompanyConstant;
import com.pangding.web.vo.system.pd.authority.*;
import com.yanzuoguang.util.base.ObjectHelper;
import com.yanzuoguang.util.exception.CodeException;
import com.yanzuoguang.util.helper.StringHelper;
import com.yanzuoguang.util.vo.PageSizeData;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.List;

/**
 * @author zhangjinyao
 */
@Service
public class RoleServiceImpl implements RoleService {

    @Autowired
    RoleDao roleDao;

    @Autowired
    UserRoleDao userRoleDaoImpl;

    @Autowired
    RoleAuthorityDao roleAuthorityDaoImpl;

    @Autowired
    AuthorityDao authorityDaoImpl;

    @Autowired
    UserDao userDaoImpl;

    /**
     * 保存RoleVo对象,并保存角色权限的对应关系
     *
     * @param webRoleReqVo
     * @return 返回角色id
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public String saveRole(WebRoleReqVo webRoleReqVo) {
        RoleVo roleVo = webRoleReqVo;
        this.checkNameExist(roleVo);
        webRoleReqVo.setId(StringHelper.getNewID());

        if (CompanyConstant.PD_COMPANY_ID.equals(roleVo.getCompanyId())) {
            roleVo.setCompanyId("");
        }
        roleDao.create(roleVo);
        List<String> authorityIds = webRoleReqVo.getAuthorityIds();
        if (authorityIds != null && !authorityIds.isEmpty()) {
            for (String authorityId : authorityIds) {
                /*查看该权限是否有上级权限*/
                findParent(authorityId, webRoleReqVo.getId());
                RoleAuthorityVo roleAuthorityRelation = new RoleAuthorityVo();
                roleAuthorityRelation.setAuthorityId(authorityId);
                roleAuthorityRelation.setRoleId(webRoleReqVo.getId());
                roleAuthorityRelation.setId(StringHelper.getNewID());
                roleAuthorityDaoImpl.create(roleAuthorityRelation);
            }
        }
        return roleVo.getId();
    }

    private void findParent(String authorityId, String roleId) {
        AuthorityGetReqVo authorityGetReqVo = new AuthorityGetReqVo();
        authorityGetReqVo.setId(authorityId);
        AuthorityVo authorityVo = authorityDaoImpl.load(authorityGetReqVo, AuthorityVo.class);
        if (null != authorityVo.getPid() && !authorityVo.getPid().isEmpty()) {
            String authorityPid = authorityVo.getPid();
            RoleAuthorityVo roleAuthorityVo = new RoleAuthorityVo();
            roleAuthorityVo.setAuthorityId(authorityPid);
            roleAuthorityVo.setRoleId(roleId);
            RoleAuthorityGetReqVo roleAuthorityGetReqVo = new RoleAuthorityGetReqVo();
            roleAuthorityGetReqVo.setRole(roleId);
            roleAuthorityGetReqVo.setAuthority(authorityPid);
            RoleAuthorityVo roleAuthorityVo1 = roleAuthorityDaoImpl.load(roleAuthorityGetReqVo, RoleAuthorityVo.class);
            if (null == roleAuthorityVo1) {
                roleAuthorityVo.setId(StringHelper.getNewID());
                roleAuthorityDaoImpl.create(roleAuthorityVo);
                findParent(authorityPid, roleId);
            }
        }
    }

    /**
     * 更新RoleVo对象,并更新角色权限的对应关系
     *
     * @param webRoleReqVo RoleDto对象
     * @return 返回角色id
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public String updateRole(WebRoleReqVo webRoleReqVo) {
        RoleVo roleVo = webRoleReqVo;
        this.checkNameExist(roleVo);

        roleDao.update(roleVo);
        RoleAuthorityGetReqVo reqVo = new RoleAuthorityGetReqVo();
        reqVo.setRole(webRoleReqVo.getId());
        List<RoleAuthorityVo> roleAuthorityList = roleAuthorityDaoImpl.loadList(reqVo, RoleAuthorityVo.class);
        if (roleAuthorityList != null && !roleAuthorityList.isEmpty()) {
            for (RoleAuthorityVo roleAuthorityRelation : roleAuthorityList) {
                RoleAuthorityVo roleAuthorityVo = new RoleAuthorityVo();
                roleAuthorityVo.setRoleId(roleAuthorityRelation.getRoleId());
                roleAuthorityVo.setAuthorityId(roleAuthorityRelation.getAuthorityId());
                roleAuthorityDaoImpl.remove(roleAuthorityVo);
            }
        }
        List<String> authorityIds = webRoleReqVo.getAuthorityIds();
        if (authorityIds != null && !authorityIds.isEmpty()) {
            for (String authorityId : authorityIds) {
                saveRoleAuthority(roleVo.getId(), authorityId);
            }
        }
        return roleVo.getId();
    }

    private void saveRoleAuthority(String roleId, String authorityId) {
        AuthorityGetReqVo getReqVo = new AuthorityGetReqVo();
        getReqVo.setId(authorityId);
        AuthorityVo authorityVo = authorityDaoImpl.load(getReqVo, AuthorityVo.class);
        if (null != authorityVo && null != authorityVo.getPid() && !authorityVo.getPid().isEmpty()) {
            saveRoleAuthority(roleId, authorityVo.getPid());
        }
        RoleAuthorityGetReqVo roleAuthorityGetReqVo = new RoleAuthorityGetReqVo();
        roleAuthorityGetReqVo.setRole(roleId);
        roleAuthorityGetReqVo.setAuthority(authorityId);
        RoleAuthorityVo existVo = roleAuthorityDaoImpl.load(roleAuthorityGetReqVo, RoleAuthorityVo.class);
        if (null == existVo) {
            RoleAuthorityVo roleAuthorityVo = new RoleAuthorityVo();
            roleAuthorityVo.setRoleId(roleId);
            roleAuthorityVo.setAuthorityId(authorityId);
            roleAuthorityVo.setId(StringHelper.getNewID());
            roleAuthorityDaoImpl.create(roleAuthorityVo);
        }
    }

    /**
     * 通过主键id查询RoleVo对象
     *
     * @param webRoleReqVo 主键
     * @return RoleVo对象
     */
    @Override
    public WebRoleResVo getRoleById(WebRoleReqVo webRoleReqVo) {
        RoleGetReqVo roleGetReqVo = new RoleGetReqVo();
        roleGetReqVo.setId(webRoleReqVo.getId());
        RoleVo roleVo = roleDao.getRoleById(roleGetReqVo);
        if (roleVo == null) {
            throw new CodeException("未找到该角色");
        }
        List<AuthorityVo> authorityList = this.getAuthorityListByRoleId(webRoleReqVo.getId());
        return this.makeResVo(roleVo, authorityList);
    }

    /**
     * 检查角色名是否已存在
     *
     * @param roleVo RoleVo对象
     */
    @Override
    public void checkNameExist(RoleVo roleVo) {
        roleDao.checkNameExist(roleVo);
    }

    /**
     * 获取角色返回信息分页对象
     *
     * @param req
     * @return 角色返回信息分页对象
     */
    @Override
    public List<WebRoleResVo> getWebRoleResVoList(ListRoleReqVo req) {
        if ("10001".equals(req.getCompanyId())) {
            req.setCompanyId("");
        }
        List<RoleVo> roleVoList = roleDao.loadList(req, RoleVo.class);
        List<WebRoleResVo> webRoleResVoList = new ArrayList();
        for (RoleVo roleVo : roleVoList) {
            WebRoleResVo resVo = new WebRoleResVo();
            ObjectHelper.writeWithFrom(resVo, roleVo);

//            List<AuthorityVo> authorityVoList = this.getAuthorityListByRoleId(roleVo.getId());
//            resVo.setAuthorityList(authorityVoList);

            CountByRoleReqVo countByRoleReqVo = new CountByRoleReqVo();
            countByRoleReqVo.setCompanyId(req.getCompanyId());
            countByRoleReqVo.setRoleId(roleVo.getId());
            List<UserVo> userVoList = count(countByRoleReqVo);
            if (null == userVoList || userVoList.isEmpty()) {
                resVo.setCount(0);
            }
            resVo.setCount(userVoList.size());

            webRoleResVoList.add(resVo);
        }

        return webRoleResVoList;
    }

    /**
     * 获取该公司该角色已分配的角色列表
     *
     * @param reqVo 封装了公司id和权限id的请求对象
     * @return 用户列表
     */
    @Override
    public List<UserVo> count(CountByRoleReqVo reqVo) {
        UserRoleGetReqVo getReqVo = new UserRoleGetReqVo();
        getReqVo.setRoleId(reqVo.getRoleId());
        List<UserRoleVo> userRoleVoList = userRoleDaoImpl.loadList(getReqVo, UserRoleVo.class);

        List<String> userIdList = new ArrayList<>();
        for (UserRoleVo userRoleVo : userRoleVoList) {
            userIdList.add(userRoleVo.getUserId());
        }
        String companyId = reqVo.getCompanyId();
        List<UserVo> userVoList = new ArrayList<>();
        for (String userId : userIdList) {
            UserGetReqVo userGetReqVo = new UserGetReqVo();
            userGetReqVo.setId(userId);
            UserVo userVo = userDaoImpl.load(userGetReqVo, UserVo.class);
            if (null != userVo && companyId.equals(userVo.getCompanyId())) {
                userVo.setPassword("");
                userVo.setRemark("");
                userVoList.add(userVo);
            }
        }
        return userVoList;
    }

    /**
     * 通过角色id查询该角色权限id列表
     *
     * @param roleId 角色id
     * @return 权限id列表
     */
    @Override
    public List<AuthorityVo> getAuthorityListByRoleId(String roleId) {
        RoleAuthorityGetReqVo reqVo = new RoleAuthorityGetReqVo();
        reqVo.setRole(roleId);
        List<AuthorityVo> roleAuthorityRelationList = roleAuthorityDaoImpl.loadList(reqVo, AuthorityVo.class);
        if (roleAuthorityRelationList != null && !roleAuthorityRelationList.isEmpty()) {
//            List<AuthorityVo> authorityList = new ArrayList(roleAuthorityRelationList.size());
//            for (RoleAuthorityVo roleAuthorityRelation : roleAuthorityRelationList) {
//                AuthorityGetReqVo authorityGetReqVo = new AuthorityGetReqVo();
//                authorityGetReqVo.setId(roleAuthorityRelation.getAuthorityId());
//                AuthorityVo authorityVo = authorityDaoImpl.load(authorityGetReqVo, AuthorityVo.class);
//                authorityList.add(authorityVo);
//            }
            return roleAuthorityRelationList;
        }
        return null;
    }

    @Override
    public String getAuthorityNameByAuthorityId(String authorityId) {
        AuthorityGetReqVo authorityGetReqVo = new AuthorityGetReqVo();
        authorityGetReqVo.setId(authorityId);
        AuthorityVo authorityVo = authorityDaoImpl.load(authorityGetReqVo, AuthorityVo.class);
        return authorityVo.getName();
    }

    @Override
    public WebRoleResVo makeResVo(RoleVo roleVo, List<AuthorityVo> authorityList) {
        WebRoleResVo resVo = new WebRoleResVo();
        resVo.setId(roleVo.getId());
        resVo.setName(roleVo.getName());
        resVo.setPid(roleVo.getPid());
        resVo.setRemark(roleVo.getRemark());
        resVo.setAuthorityList(authorityList);
        return resVo;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteRole(RoleVo roleVo) {

        roleDao.remove(roleVo);

        UserRoleGetReqVo userRoleGetReqVo = new UserRoleGetReqVo();
        userRoleGetReqVo.setRoleId(roleVo.getId());
        List<UserRoleVo> userRoleList = userRoleDaoImpl.getUserRoleList(userRoleGetReqVo);
        if (userRoleList != null && !userRoleList.isEmpty()) {
            for (UserRoleVo userRoleRelation : userRoleList) {
                userRoleDaoImpl.remove(userRoleRelation);
            }
        }

        RoleAuthorityGetReqVo reqVo = new RoleAuthorityGetReqVo();
        reqVo.setRoleId(roleVo.getId());
        List<RoleAuthorityVo> roleAuthorityList = roleAuthorityDaoImpl.loadList(reqVo, RoleAuthorityVo.class);
        if (roleAuthorityList != null && !roleAuthorityList.isEmpty()) {
            for (RoleAuthorityVo roleAuthority : roleAuthorityList) {
                roleAuthorityDaoImpl.remove(roleAuthority);
            }
        }

        RoleGetReqVo roleGetReqVo = new RoleGetReqVo();
        roleGetReqVo.setId(roleVo.getId());
        List<RoleVo> childList = roleDao.getChild(roleGetReqVo);
        for (RoleVo child : childList) {
            this.deleteRole(child);
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void delete(WebRoleReqVo webRoleReqVo) {
        RoleVo roleVo = this.getRoleById(webRoleReqVo);
        this.deleteRole(roleVo);
    }

    @Override
    public PageSizeData<UserVo> countByRole(CountByRoleReqVo reqVo) {
        return userDaoImpl.countByRole(reqVo);
    }

    @Override
    public void add(AddReqVo reqVo) {
        RoleAuthorityVo roleAuthorityVo = new RoleAuthorityVo();
        ObjectHelper.writeWithFrom(roleAuthorityVo, reqVo);
        roleAuthorityVo.setId(StringHelper.getNewID());
        roleAuthorityDaoImpl.create(roleAuthorityVo);
    }
}
