package com.pangding.web.authority.controller;

import com.pangding.web.authority.service.PangdingPartnerService;
import com.pangding.web.authority.vo.reqvo.*;
import com.pangding.web.authority.vo.resvo.*;
import com.pangding.web.vo.system.pd.company.CompanyBankCardVo;
import com.pangding.web.vo.system.req.company.CompanyBankCardReqVo;
import com.yanzuoguang.util.helper.CheckerHelper;
import com.yanzuoguang.util.vo.ResponseResult;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.util.List;

/**
 * @Author zhangjinyao
 * @create 2019/6/19 15:26
 */
@RestController
@RequestMapping(value = "/partner")
public class PangdingPartnerController {
    @Autowired
    PangdingPartnerService pangdingPartnerServiceImpl;

    /**
     * 胖丁伙伴首页
     * @param reqVo 封装了公司id的请求对象
     * @return
     */
    @PostMapping(value = "/home",produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseResult<HomeResVo> home(@RequestBody HomeReqVo reqVo){
        CheckerHelper.newInstance()
                .notBlankCheck("companyId",reqVo.getCompanyId())
                .checkException();
        return ResponseResult.result(pangdingPartnerServiceImpl.home(reqVo));
    }

    /**
     * 通过公司id获取该公司的可提现余额以及默认提现银行卡
     * @param reqVo 封装了公司id的请求对象
     * @return
     */
    @PostMapping(value = "/withdrawal",produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseResult<MobileWithdrawalResVo> withdrawal(@RequestBody MobileWithdrawalReqVo reqVo){
        CheckerHelper.newInstance()
                .notBlankCheck("companyId",reqVo.getCompanyId())
                .notBlankCheck("withdrawTotalCash",reqVo.getWithdrawTotalCash())
                .checkException();
        return ResponseResult.result(pangdingPartnerServiceImpl.withdrawal(reqVo));
    }

    /**
     * 通过公司id获取该公司银行卡列表（不分页）
     * @param reqVo 封装了公司id的请求对象
     * @return
     */
    @PostMapping(value = "/bankCardList",produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseResult<List<CompanyBankCardVo>> bankCardList(@RequestBody MobileBankCardListReqVo reqVo){
        CheckerHelper.newInstance()
                .notBlankCheck("companyId",reqVo.getCompanyId())
                .checkException();
        return ResponseResult.result(pangdingPartnerServiceImpl.bankCardList(reqVo));
    }

    /**
     * 保存/更新银行卡信息，并调用第三方接口发送手机验证码
     * @param reqVo 公司银行卡表信息
     * @return 银行卡表主键
     */
    @PostMapping(value = "/saveBank",produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseResult<String> saveBank(@RequestBody CompanyBankCardReqVo reqVo){
        checkBank(reqVo);
        return ResponseResult.result(pangdingPartnerServiceImpl.saveBank(reqVo));
    }

    private void checkBank(CompanyBankCardReqVo reqVo){
        CheckerHelper.newInstance()
                .notBlankCheck("bankCardCode",reqVo.getBankCardCode())
                .notNumberCheck("bankCardCode",reqVo.getBankCardCode())
                .notBlankCheck("bankName",reqVo.getBankName())
                .notBlankCheck("name",reqVo.getName())
                .notBlankCheck("idCard",reqVo.getIdCard())
                .notBlankCheck("companyId",reqVo.getCompanyId())
                .checkIdCard("idCard",reqVo.getIdCard())
                .notBlankCheck("phone",reqVo.getPhone())
                .checkPhoneNo("phone",reqVo.getPhone())
                .checkException();
    }

    /**
     * 提交银行卡信息，验证手机验证码，并更新银行卡状态
     * @param reqVo 银行卡表数据+验证码
     * @return 目前返回验证码
     */
    @PostMapping(value = "/bankSubmit",produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseResult<CompanyBankCardVo> bankSubmit(@RequestBody BankSubmitReqVo reqVo) {
        CheckerHelper.newInstance()
                .notBlankCheck("bankCardId", reqVo.getId())
                .notBlankCheck("verifyCode", reqVo.getVerifyCode())
                .checkException();
        return ResponseResult.result(pangdingPartnerServiceImpl.bankSubmit(reqVo));
    }

    /**
     * 调用支付服务解绑银行卡，并更新银行卡状态
     * @param reqVo 封装了银行卡id的请求对象
     * @return
     */
    @PostMapping(value = "/deleteBank",produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseResult deleteBank(@RequestBody CompanyBankCardVo reqVo){
        CheckerHelper.newInstance()
                .notBlankCheck("companyBankCardId",reqVo.getId())
                .checkException();
        pangdingPartnerServiceImpl.deleteBank(reqVo);
        return new ResponseResult();
    }

    /**
     * 设置该银行卡为默认支付，并取消其他银行卡的默认
     * @param reqVo 封装了银行卡id的请求对象
     * @return
     */
    @PostMapping(value = "setDefault",produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseResult setDefault(@RequestBody CompanyBankCardVo reqVo){
        CheckerHelper.newInstance()
                .notBlankCheck("bankCardId",reqVo.getId())
                .checkException();
        pangdingPartnerServiceImpl.setDefault(reqVo);
        return new ResponseResult();
    }

    /**
     * 我的管理页面：通过userId获取用户表，公司表，公司银行卡表数据
     * @param reqVo 封装了userId的请求对象
     * @return
     */
    @PostMapping(value = "/management",produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseResult<ManagementResVo> management(@RequestBody ManagementReqVo reqVo){
        CheckerHelper.newInstance()
                .notBlankCheck("userId",reqVo.getUserId())
                .checkException();
        return ResponseResult.result(pangdingPartnerServiceImpl.management(reqVo));
    }

    /**
     * 根据公司id获取该公司今日用户沉淀量，总沉淀量，沉淀用户列表
     * @param reqVo 封装了公司id的请求对象
     * @return
     */
    @PostMapping(value = "/precipitation",produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseResult<PrecipitationResVo> precipitation(@RequestBody PrecipitationReqVo reqVo){
        CheckerHelper.newInstance()
                .notBlankCheck("companyId",reqVo.getCompanyId())
                .checkException();
        return ResponseResult.result(pangdingPartnerServiceImpl.precipitation(reqVo));
    }

    /**
     * 根据公司id返回该公司二维码url，若没有则创建
     * @param reqVo 封装了公司id和选项（限制为永久）的请求对象
     * @return
     */
    @PostMapping(value = "/qrCodeUrl",produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseResult<String> qrCodeUrl(@RequestBody GenerateQrCodeUrlReqVo reqVo){
        CheckerHelper.newInstance()
                .notBlankCheck("companyId",reqVo.getCompanyId())
                .notBlankCheck("select",reqVo.getSelect())
                .checkException();
        return ResponseResult.result(pangdingPartnerServiceImpl.qrCodeUrl(reqVo));
    }

    /**
     * 根据公司id获取该公司的邀请码，若没有则创建
     * @param reqVo 封装了公司id的请求对象
     * @return
     */
    @PostMapping(value = "/invitation",produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseResult<String> invitation(@RequestBody InvitationReqVo reqVo){
        CheckerHelper.newInstance()
                .notBlankCheck("companyId",reqVo.getCompanyId())
                .checkException();
        return ResponseResult.result(pangdingPartnerServiceImpl.invitation(reqVo));
    }

    /**
     * 邀请商家列表
     * @param reqVo
     * @return
     */
    @PostMapping(value = "/inviteMembers",produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseResult<InviteMembersResVo> inviteMembers(@RequestBody InviteMembersReqVo reqVo){
        CheckerHelper.newInstance()
                .notBlankCheck("invitationCompanyId",reqVo.getInvitationCompanyId())
                .checkException();
        return ResponseResult.result(pangdingPartnerServiceImpl.inviteMembers(reqVo));
    }
}
