package com.pangding.web.authority.service.impl;

import com.pangding.web.authority.dao.CompanyBankCardDao;
import com.pangding.web.authority.dao.CompanyDao;
import com.pangding.web.authority.dao.CompanyExtendDao;
import com.pangding.web.authority.dao.UserDao;
import com.pangding.web.authority.service.CompanyService;
import com.pangding.web.authority.service.UserService;
import com.pangding.web.authority.util.CompanyBankCardConstant;
import com.pangding.web.authority.vo.CompanyBankCardVo;
import com.pangding.web.authority.vo.CompanyExtendVo;
import com.pangding.web.authority.vo.CompanyVo;
import com.pangding.web.authority.vo.UserVo;
import com.pangding.web.authority.vo.reqvo.*;
import com.pangding.web.authority.vo.resvo.CompanyResVo;
import com.pangding.web.authority.vo.resvo.DetailResVo;
import com.yanzuoguang.util.base.ObjectHelper;
import com.yanzuoguang.util.exception.CodeException;
import com.yanzuoguang.util.helper.DateHelper;
import com.yanzuoguang.util.helper.StringHelper;
import com.yanzuoguang.util.vo.PageSizeData;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;

/**
 * @author Administrator
 */
@Service
public class CompanyServiceImpl implements CompanyService{

    @Autowired
    private CompanyDao companyDao;
    @Autowired
    private CompanyExtendDao companyExtendDaoImpl;
    @Autowired
    private CompanyBankCardDao companyBankCardDao;
    @Autowired
    private UserService userServiceImpl;
    @Autowired
    private UserDao userDaoImpl;

    /**
     * 修改公司信息
     * @param companyVo
     */
    @Override
    public void updateCompanyStatus(CompanyVo companyVo){
        CompanyVo company = companyDao.load(companyVo.getId(), CompanyVo.class);
        ObjectHelper.writeWithFrom(company, companyVo);
        companyDao.update(company);
    }

    /**
     * 分页查询公司信息
     * @param reqVo 继承了分页对象
     * @return 公司列表分页对象
     */
    @Override
    public PageSizeData<CompanyVo> list(ListCompanyReqVo reqVo) {
        PageSizeData<CompanyVo> pageSizeData = companyDao.getCompanyList(reqVo);
        if (null == pageSizeData || null == pageSizeData.getList() || pageSizeData.getList().isEmpty()){
            throw new CodeException("暂无公司信息");
        }
        return pageSizeData;
    }

    /**
     * 胖丁查询用户，公司和公司扩展表信息以及银行卡列表
     * @param reqVo 包含公司id的请求对象
     * @return 封装了用户，公司，公司扩展表和银行卡列表信息的对象
     */
    @Override
    public CompanyResVo pdGetDetail(CompanyGetReqVo reqVo) {
        CompanyResVo resVo = new CompanyResVo();
        /*获取公司表信息*/
        CompanyVo companyVo = companyDao.load(reqVo,CompanyVo.class);
        if (null != companyVo){
            resVo.setCompanyVo(companyVo);
        }
        /*获取公司扩展表信息*/
        CompanyExtendGetReqVo companyExtendGetReqVo = new CompanyExtendGetReqVo();
        companyExtendGetReqVo.setId(reqVo.getId());
        CompanyExtendVo companyExtendVo = companyExtendDaoImpl.load(companyExtendGetReqVo,CompanyExtendVo.class);
        if (null != companyExtendVo){
            resVo.setCompanyExtendVo(companyExtendVo);
        }
        /*获取公司银行卡列表*/
        CompanyBankCardGetReqVo companyBankCardGetReqVo = new CompanyBankCardGetReqVo();
        companyBankCardGetReqVo.setCompanyId(reqVo.getId());
        List<CompanyBankCardVo> companyBankCardVoList = companyBankCardDao.loadList(companyBankCardGetReqVo,CompanyBankCardVo.class);
        if (null != companyBankCardVoList && !companyBankCardVoList.isEmpty()){
            resVo.setBankCardVoList(companyBankCardVoList);
        }
        /*获取用户表信息*/
        UserGetReqVo userGetReqVo = new UserGetReqVo();
        userGetReqVo.setCompanyId(reqVo.getId());
        UserVo userVo = userDaoImpl.load(userGetReqVo,UserVo.class);
        if (null != userVo){
            userVo.setPassword("");
            userVo.setRemark("");
            resVo.setUserVo(userVo);
        }
        return resVo;
    }

    /**
     * 商户查看详情，根据公司id返回用户信息和公司主表信息
     * @param reqVo 包含公司id的请求对象
     * @return 封装了用户，公司主表信息的对象
     */
    @Override
    public DetailResVo getDetail(CompanyGetReqVo reqVo) {
        DetailResVo resVo = new DetailResVo();
        /*查询公司主表信息*/
        CompanyVo companyVo = companyDao.load(reqVo,CompanyVo.class);
        if (null != companyVo){
            resVo.setCompanyVo(companyVo);
        }
        /*查询用户信息*/
        UserGetReqVo userGetReqVo = new UserGetReqVo();
        userGetReqVo.setCompanyId(reqVo.getId());
        UserVo userVo = userDaoImpl.load(userGetReqVo,UserVo.class);
        if (null != userVo){
            userVo.setPassword("");
            userVo.setRemark("");
            resVo.setUserVo(userVo);
        }
        return resVo;
    }

    /**
     * 新增/修改公司信息
     * @param reqVo 公司以及公司扩展表信息
     * @return 公司id
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public CompanyVo saveCompany(CompanyAndCompanyExtendSaveReqVo reqVo) {
        CompanyVo companyVo = new CompanyVo();
        ObjectHelper.writeWithFrom(companyVo,reqVo);

        CompanyExtendVo companyExtendVo = new CompanyExtendVo();
        ObjectHelper.writeWithFrom(companyExtendVo,reqVo);

        String id = StringHelper.getNewID();
        /*若id为空，则新建，若不为空，则更新,且更新修改时间*/
        /*新增*/
        if (null == reqVo.getId() || reqVo.getId().isEmpty()){
            /*保存公司表*/
            /*此处需要生成二维码源码*/
            companyVo.setId(id);
            companyDao.create(companyVo);
            /*保存公司扩展表*/
            companyExtendVo.setId(id);
            companyExtendDaoImpl.create(companyExtendVo);
        }
        /*修改*/
        else{
            /*更新公司表*/
            companyVo.setUpdateTime(DateHelper.getNow());
            companyDao.update(companyVo);
            /*更新公司扩展表*/
            companyExtendDaoImpl.update(companyExtendVo);
        }
        return companyVo;
    }

    /**
     * 保存/更新银行卡信息，并调用第三方接口发送手机验证码
     *
     * @param reqVo 公司银行卡表信息
     * @return 银行卡表主键
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public String saveBank(CompanyBankCardVo reqVo) {
        String id = StringHelper.getNewID();

        /*若该卡存在，则修改*/
        CompanyBankCardGetReqVo getReqVo = new CompanyBankCardGetReqVo();
        getReqVo.setId(reqVo.getId());
        CompanyBankCardVo companyBankCardVo = companyBankCardDao.load(getReqVo,CompanyBankCardVo.class);
        if (null != companyBankCardVo){
            companyBankCardDao.update(reqVo);
        }
        /*若卡不存在，则新增*/
        else{
            reqVo.setId(id);
            BankCardListReqVo reqVo1 = new BankCardListReqVo();
            PageSizeData<CompanyBankCardVo> companyBankCardDaoBankCardList = companyBankCardDao.getBankCardList(reqVo1);

            /*若该银行卡是第一张银行卡，则设为默认*/
            if (null == companyBankCardDaoBankCardList || companyBankCardDaoBankCardList.getList().isEmpty()){
                reqVo.setIsDefault(CompanyBankCardConstant.IS_DEFAULT);
            }
            companyBankCardDao.create(reqVo);
        }

        /*此处需要调用第三方接口发送手机验证码*/
        return reqVo.getId();
    }

    @Override
    public void deleteBank(CompanyBankCardVo reqVo) {
        companyBankCardDao.remove(reqVo);
    }

    @Override
    public String updateBank(CompanyBankCardVo reqVo) {
        companyBankCardDao.update(reqVo);
        return reqVo.getId();
    }

    @Override
    public void setDefault(CompanyBankCardVo reqVo) {
        /*获取该银行卡的公司id*/
        CompanyBankCardVo companyBankCardVo = companyBankCardDao.load(reqVo,CompanyBankCardVo.class);
        String companyId = companyBankCardVo.getCompanyId();
        /*现将该公司其他银行卡设为非默认*/
        CompanyBankCardSetNotDefaultReqVo setNotDefaultReqVo = new CompanyBankCardSetNotDefaultReqVo();
        setNotDefaultReqVo.setCompanyId(companyId);
        companyBankCardDao.setNotDefault(setNotDefaultReqVo);
        /*将该卡设置为默认银行卡并更新*/
        companyBankCardVo.setIsDefault(CompanyBankCardConstant.IS_DEFAULT);
        companyBankCardDao.update(companyBankCardVo);
    }

    /**
     * 提交银行卡信息，验证手机验证码，并更新银行卡状态
     * @param reqVo 银行卡表数据+验证码
     * @return 目前返回验证码
     */
    @Override
    public CompanyBankCardVo bankSubmit(BankSubmitReqVo reqVo) {
        if (null != reqVo.getVerifyCode() || reqVo.getVerifyCode().isEmpty()){
            /*此处应该将请求数据发送到第三方接口，获取到的返回值用于更新银行卡状态*/
        }

        CompanyBankCardVo companyBankCardVo = new CompanyBankCardVo();
        ObjectHelper.writeWithFrom(companyBankCardVo,reqVo);
        return companyBankCardVo;
    }

    /**
     * 获取我的二维码url
     * @param reqVo 公司扩展表请求对象
     * @return 公司二维码url
     */
    @Override
    public String getQrCodeUrl(CompanyExtendGetReqVo reqVo) {
        CompanyExtendVo companyExtendVo = companyExtendDaoImpl.load(reqVo,CompanyExtendVo.class);
        if (null == companyExtendVo.getQrCodeUrl() || companyExtendVo.getQrCodeUrl().isEmpty()){
            throw new CodeException("未拥有二维码url，请先添加");
        }
        return companyExtendVo.getQrCodeUrl();
    }

    /**
     * 根据二维码源码调用微信接口获取二维码url
     * @param reqVo 封装了公司id，页面选项的请求对象
     * @return 二维码url，目前没有调用微信接口，仅返回null
     */
    @Override
    public String createQrCodeUrl(CreateQrCodeUrlReqVo reqVo) {
        CompanyExtendGetReqVo reqVo1 = new CompanyExtendGetReqVo();
        reqVo1.setId(reqVo.getCompanyId());
        CompanyExtendVo companyExtendVo = companyExtendDaoImpl.load(reqVo1,CompanyExtendVo.class);
        String qrCode = companyExtendVo.getQrCode();
        String select = reqVo.getSelect();
        /*此处应调用微信接口传入qrCode和select获取qrCodeUrl*/
        return null;
    }

    /**
     * 获取银行卡列表
     * @param reqVo 继承了分页对象，含公司id
     * @return 银行卡列表分页对象
     */
    @Override
    public PageSizeData<CompanyBankCardVo> getBankCardList(BankCardListReqVo reqVo) {
        PageSizeData<CompanyBankCardVo> companyBankCardVoPageSizeData = companyBankCardDao.getBankCardList(reqVo);
        if (companyBankCardVoPageSizeData == null || companyBankCardVoPageSizeData.getList().isEmpty()) {
            throw new CodeException("暂无银行卡，请先添加");
        }
        return companyBankCardVoPageSizeData;
    }

    /**
     * 添加用户
     * @param reqVo 封装了用户信息和用户角色信息的请求对象
     * @return 用户id
     */
    @Override
    public String saveUser(WebUserReqVo reqVo) {
        return userServiceImpl.saveUser(reqVo);
    }
}
