package com.pangding.web.authority.service.impl;

import com.pangding.web.authority.dao.RoleDao;
import com.pangding.web.authority.dao.RoleAuthorityDao;
import com.pangding.web.authority.dao.AuthorityDao;
import com.pangding.web.authority.dao.UserRoleDao;
import com.pangding.web.authority.vo.reqvo.*;
import com.pangding.web.authority.service.RoleService;
import com.pangding.web.authority.vo.*;
import com.pangding.web.authority.vo.resvo.WebRoleResVo;
import com.yanzuoguang.util.exception.CodeException;
import com.yanzuoguang.util.helper.StringHelper;
import com.yanzuoguang.util.vo.PageSizeData;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.List;

/**
 * @author zhangjinyao
 */
@Service
public class RoleServiceImpl implements RoleService {

    @Autowired
    RoleDao roleDao;

    @Autowired
    UserRoleDao userRoleDaoImpl;

    @Autowired
    RoleAuthorityDao roleAuthorityDaoImpl;

    @Autowired
    AuthorityDao authorityDaoImpl;

    /**
     * 保存RoleVo对象,并保存角色权限的对应关系
     *
     * @param webRoleReqVo
     * @return 返回角色id
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public String saveRole(WebRoleReqVo webRoleReqVo) {
        RoleVo roleVo = webRoleReqVo;
        this.checkNameExist(roleVo);
        webRoleReqVo.setId(StringHelper.getNewID());

        roleDao.create(roleVo);
        List<String> authorityIds = webRoleReqVo.getAuthorityIds();
        if (authorityIds != null && !authorityIds.isEmpty()){
            for (String authorityId : authorityIds) {
                RoleAuthorityVo roleAuthorityRelation = new RoleAuthorityVo();
                roleAuthorityRelation.setAuthorityId(authorityId);
                roleAuthorityRelation.setRoleId(webRoleReqVo.getId());
                roleAuthorityRelation.setId(StringHelper.getNewID());
                roleAuthorityDaoImpl.create(roleAuthorityRelation);
            }
        }
        return roleVo.getId();
    }

    /**
     * 更新RoleVo对象,并更新角色权限的对应关系
     *
     * @param webRoleReqVo RoleDto对象
     * @return 返回角色id
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public String updateRole(WebRoleReqVo webRoleReqVo) {
        RoleVo roleVo = webRoleReqVo;
        this.checkNameExist(roleVo);

        roleDao.update(roleVo);
        RoleAuthorityGetReqVo reqVo = new RoleAuthorityGetReqVo();
        reqVo.setRoleId(webRoleReqVo.getId());
        List<RoleAuthorityVo> roleAuthorityList = roleAuthorityDaoImpl.loadList(reqVo,RoleAuthorityVo.class);
        if (roleAuthorityList != null && !roleAuthorityList.isEmpty()){
            for (RoleAuthorityVo roleAuthorityRelation:roleAuthorityList) {
                roleAuthorityDaoImpl.remove(roleAuthorityRelation);
            }
        }
        List<String> authorityIds = webRoleReqVo.getAuthorityIds();
        if (authorityIds != null && !authorityIds.isEmpty()){
            for (String authorityId : authorityIds) {
                RoleAuthorityVo roleAuthorityRelation = new RoleAuthorityVo();
                roleAuthorityRelation.setAuthorityId(authorityId);
                roleAuthorityRelation.setRoleId(webRoleReqVo.getId());
                roleAuthorityRelation.setId(StringHelper.getNewID());
                roleAuthorityDaoImpl.create(roleAuthorityRelation);
            }
        }
        return roleVo.getId();
    }

    /**
     * 通过主键id查询RoleVo对象
     *
     * @param webRoleReqVo 主键
     * @return RoleVo对象
     */
    @Override
    public WebRoleResVo getRoleById(WebRoleReqVo webRoleReqVo) {
        RoleGetReqVo roleGetReqVo = new RoleGetReqVo();
        roleGetReqVo.setId(webRoleReqVo.getId());
        RoleVo roleVo = roleDao.getRoleById(roleGetReqVo);
        if (roleVo == null){
            throw new CodeException("未找到该角色");
        }
        List<AuthorityVo> authorityList = this.getAuthorityListByRoleId(webRoleReqVo.getId());
        return this.makeResVo(roleVo,authorityList);
    }

    /**
     * 检查角色名是否已存在
     *
     * @param roleVo RoleVo对象
     */
    @Override
    public void checkNameExist(RoleVo roleVo){
        roleDao.checkNameExist(roleVo);
    }

    /**
     * 获取RoleVo对象列表
     *
     * @param req
     * @return RoleVo对象列表分页对象
     */
    @Override
    public PageSizeData<RoleVo> getRoleVoList(ListRoleReqVo req) {

        PageSizeData<RoleVo> roleVoPageSizeData = roleDao.getRoleVoList(req);
        if (roleVoPageSizeData.getList() == null || roleVoPageSizeData.getList().isEmpty()){
            throw new CodeException("没有更多权限");
        }
        return roleVoPageSizeData;
    }

    /**
     * 获取角色返回信息分页对象
     *
     * @param req
     * @return 角色返回信息分页对象
     */
    @Override
    public PageSizeData<WebRoleResVo> getWebRoleResVoList(ListRoleReqVo req) {
        PageSizeData<RoleVo> roleVoPageSizeData = this.getRoleVoList(req);
        PageSizeData<WebRoleResVo> webRoleResVoPageSizeData = new PageSizeData();
        List<RoleVo> roleVoList = roleVoPageSizeData.getList();
        List<WebRoleResVo> webRoleResVoList = new ArrayList(roleVoList.size());
        for (RoleVo roleVo:roleVoList) {
            WebRoleResVo resVo = new WebRoleResVo();
            List<AuthorityVo> authorityVoList = this.getAuthorityListByRoleId(roleVo.getId());

            resVo.setAuthorityList(authorityVoList);
            resVo.setId(roleVo.getId());
            resVo.setName(roleVo.getName());
            resVo.setPid(roleVo.getPid());
            resVo.setRemark(roleVo.getRemark());

            webRoleResVoList.add(resVo);
        }
        webRoleResVoPageSizeData.setPageTotal(roleVoPageSizeData.getPageTotal());
        webRoleResVoPageSizeData.setList(webRoleResVoList);
        webRoleResVoPageSizeData.setPageIndex(roleVoPageSizeData.getPageIndex());
        webRoleResVoPageSizeData.setPageSize(roleVoPageSizeData.getPageSize());

        return webRoleResVoPageSizeData;
    }

    /**
     * 通过角色id查询该角色权限id列表
     *
     * @param roleId 角色id
     * @return 权限id列表
     */
    @Override
    public List<AuthorityVo> getAuthorityListByRoleId(String roleId) {
        RoleAuthorityGetReqVo reqVo = new RoleAuthorityGetReqVo();
        reqVo.setRoleId(roleId);
        List<RoleAuthorityVo> roleAuthorityRelationList = roleAuthorityDaoImpl.loadList(reqVo,RoleAuthorityVo.class);
        if (roleAuthorityRelationList != null && !roleAuthorityRelationList.isEmpty()){
            List<AuthorityVo> authorityList = new ArrayList(roleAuthorityRelationList.size());
            for (RoleAuthorityVo roleAuthorityRelation:roleAuthorityRelationList) {
                AuthorityGetReqVo authorityGetReqVo = new AuthorityGetReqVo();
                authorityGetReqVo.setId(roleAuthorityRelation.getAuthorityId());
                AuthorityVo authorityVo = authorityDaoImpl.load(authorityGetReqVo,AuthorityVo.class);
                authorityList.add(authorityVo);
            }
            return authorityList;
        }
        return null;
    }

    @Override
    public String getAuthorityNameByAuthorityId(String authorityId) {
        AuthorityGetReqVo authorityGetReqVo = new AuthorityGetReqVo();
        authorityGetReqVo.setId(authorityId);
        AuthorityVo authorityVo = authorityDaoImpl.load(authorityGetReqVo,AuthorityVo.class);
        return authorityVo.getName();
    }

    @Override
    public WebRoleResVo makeResVo(RoleVo roleVo, List<AuthorityVo> authorityList) {
        WebRoleResVo resVo = new WebRoleResVo();
        resVo.setId(roleVo.getId());
        resVo.setName(roleVo.getName());
        resVo.setPid(roleVo.getPid());
        resVo.setRemark(roleVo.getRemark());
        resVo.setAuthorityList(authorityList);
        return resVo;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteRole(RoleVo roleVo) {

        roleDao.remove(roleVo);

        UserRoleGetReqVo userRoleGetReqVo = new UserRoleGetReqVo();
        userRoleGetReqVo.setRoleId(roleVo.getId());
        List<UserRoleVo> userRoleList = userRoleDaoImpl.getUserRoleList(userRoleGetReqVo);
        if (userRoleList != null && !userRoleList.isEmpty()){
            for (UserRoleVo userRoleRelation:userRoleList) {
                userRoleDaoImpl.remove(userRoleRelation);
            }
        }

        RoleAuthorityGetReqVo reqVo = new RoleAuthorityGetReqVo();
        reqVo.setRoleId(roleVo.getId());
        List<RoleAuthorityVo> roleAuthorityList = roleAuthorityDaoImpl.loadList(reqVo,RoleAuthorityVo.class);
        if (roleAuthorityList != null && !roleAuthorityList.isEmpty()){
            for (RoleAuthorityVo roleAuthority:roleAuthorityList) {
                roleAuthorityDaoImpl.remove(roleAuthority);
            }
        }

        RoleGetReqVo roleGetReqVo = new RoleGetReqVo();
        roleGetReqVo.setId(roleVo.getId());
        List<RoleVo> childList = roleDao.getChild(roleGetReqVo);
        for (RoleVo child:childList) {
            this.deleteRole(child);
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void delete(WebRoleReqVo webRoleReqVo) {
        RoleVo roleVo = this.getRoleById(webRoleReqVo);
        this.deleteRole(roleVo);
    }
}
